<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\HomeController;
use App\Http\Controllers\ProductController;
use App\Http\Controllers\CartController;
use App\Http\Controllers\OrderController;
use App\Http\Controllers\AuthController;
use App\Http\Controllers\CurrencyController;
use App\Http\Controllers\Admin\SliderController;
use App\Http\Controllers\Admin\DashboardController;
use App\Http\Controllers\Admin\SettingsController;
use App\Http\Controllers\Admin\ProductController as AdminProductController;
use App\Http\Controllers\Admin\OrderController as AdminOrderController;
use App\Http\Controllers\Admin\CategoryController;
use App\Http\Controllers\Admin\UserController;
use App\Http\Controllers\Admin\AnalyticsController;
use App\Http\Controllers\ProfileController;
use App\Http\Controllers\Store\StoreController;
use App\Http\Controllers\CustomerDashboardController;
use App\Http\Controllers\NewsletterController;
use App\Http\Controllers\PageController;

/*
|--------------------------------------------------------------------------
| Web Routes
|--------------------------------------------------------------------------
|
| Here is where you can register web routes for your application. These
| routes are loaded by the RouteServiceProvider within a group which
| contains the "web" middleware group. Now create something great!
|
*/

// Home
Route::get('/', [HomeController::class, 'index'])->name('home');

// Newsletter
Route::post('/newsletter/subscribe', [NewsletterController::class, 'subscribe'])->name('newsletter.subscribe');

// Pages
Route::get('/about-us', [PageController::class, 'about'])->name('pages.about');
Route::get('/privacy-policy', [PageController::class, 'privacy'])->name('pages.privacy');
Route::get('/terms-conditions', [PageController::class, 'terms'])->name('pages.terms');
Route::get('/contact-us', [PageController::class, 'contact'])->name('pages.contact');
Route::get('/returns-exchanges', [PageController::class, 'returns'])->name('pages.returns');
Route::get('/faqs', [PageController::class, 'faqs'])->name('pages.faqs');
Route::get('/size-guide', [PageController::class, 'sizeGuide'])->name('pages.size-guide');

// Products
Route::get('/products', [ProductController::class, 'index'])->name('products.index');
Route::get('/products/{slug}', [ProductController::class, 'show'])->name('products.show');
Route::get('/products/search/ajax', [ProductController::class, 'search'])->name('products.search');

// Authentication
Route::get('/login', [AuthController::class, 'showLoginForm'])->name('login');
Route::post('/login', [AuthController::class, 'login']);
Route::get('/register', [AuthController::class, 'showRegisterForm'])->name('register');
Route::post('/register', [AuthController::class, 'register']);
Route::get('/verify-email', [AuthController::class, 'showVerifyEmailForm'])->name('verify.email');
Route::post('/verify-email', [AuthController::class, 'verifyEmail']);
Route::post('/resend-otp', [AuthController::class, 'resendOtp'])->name('resend.otp');
Route::post('/logout', [AuthController::class, 'logout'])->name('logout');

// Password Reset
Route::get('/forgot-password', [AuthController::class, 'showForgotPasswordForm'])->name('password.forgot');
Route::post('/forgot-password', [AuthController::class, 'sendPasswordResetOtp'])->name('password.email');
Route::get('/reset-password', [AuthController::class, 'showResetPasswordForm'])->name('password.reset');
Route::post('/reset-password', [AuthController::class, 'resetPassword'])->name('password.update');
Route::post('/resend-password-reset', [AuthController::class, 'resendPasswordResetOtp'])->name('password.resend');

// Cart (available for guests)
Route::get('/cart', [CartController::class, 'index'])->name('cart.index');
Route::post('/cart', [CartController::class, 'store'])->name('cart.store');
Route::put('/cart/{id}', [CartController::class, 'update'])->name('cart.update');
Route::delete('/cart/{id}', [CartController::class, 'destroy'])->name('cart.destroy');

// Checkout (available for guests)
Route::get('/checkout', [OrderController::class, 'checkout'])->name('checkout');
Route::post('/orders', [OrderController::class, 'store'])->name('orders.store');

// Payment
Route::post('/pay', [App\Http\Controllers\PaymentController::class, 'redirectToGateway'])->name('pay');
Route::get('/payment/callback', [App\Http\Controllers\PaymentController::class, 'handleGatewayCallback'])->name('payment.callback');

// User Profile (requires authentication)
Route::middleware('auth')->group(function () {
    Route::get('/profile', [ProfileController::class, 'index'])->name('profile.index');
    Route::put('/profile', [ProfileController::class, 'update'])->name('profile.update');
});

// Customer Dashboard (requires authentication)
Route::middleware('auth')->group(function () {
    // Dashboard
    Route::get('/dashboard', [CustomerDashboardController::class, 'index'])->name('customer.dashboard');
    
    // Wishlist
    Route::get('/wishlist', [CustomerDashboardController::class, 'wishlist'])->name('customer.wishlist');
    Route::post('/wishlist/{productId}', [CustomerDashboardController::class, 'addToWishlist'])->name('customer.wishlist.add');
    Route::delete('/wishlist/{productId}', [CustomerDashboardController::class, 'removeFromWishlist'])->name('customer.wishlist.remove');
    
    // Addresses
    Route::get('/addresses', [CustomerDashboardController::class, 'addresses'])->name('customer.addresses');
    Route::get('/addresses/{id}/edit', [CustomerDashboardController::class, 'editAddress'])->name('customer.addresses.edit');
    Route::post('/addresses', [CustomerDashboardController::class, 'storeAddress'])->name('customer.addresses.store');
    Route::put('/addresses/{id}', [CustomerDashboardController::class, 'updateAddress'])->name('customer.addresses.update');
    Route::delete('/addresses/{id}', [CustomerDashboardController::class, 'deleteAddress'])->name('customer.addresses.delete');
    Route::post('/addresses/{id}/set-default', [CustomerDashboardController::class, 'setDefaultAddress'])->name('customer.addresses.set-default');
});

// Orders
Route::get('/orders/{id}', [OrderController::class, 'show'])->name('orders.show'); // Allow guests to view their order confirmation

// Order history (requires authentication)
Route::middleware('auth')->group(function () {
    Route::get('/orders', [OrderController::class, 'index'])->name('orders.index');
});

// Currency
Route::post('/currency/change', [CurrencyController::class, 'change'])->name('currency.change');

// Analytics Click Tracking (Public route for tracking - no auth required)
Route::post('/analytics/click', [AnalyticsController::class, 'trackClick'])->name('analytics.trackClick');

// Admin Routes
Route::middleware(['auth', 'admin'])->prefix('admin')->name('admin.')->group(function () {
    Route::get('/dashboard', [DashboardController::class, 'index'])->name('dashboard');
    Route::get('/stores', [\App\Http\Controllers\Admin\StoreController::class, 'index'])->name('stores.index');
    Route::get('/download-qrcode', [DashboardController::class, 'downloadQrCode'])->name('download-qrcode');
    Route::post('sliders/banner-settings', [SliderController::class, 'updateBannerSettings'])->name('sliders.banner-settings');
    Route::resource('sliders', SliderController::class);
    Route::resource('products', AdminProductController::class);
    Route::resource('categories', CategoryController::class);
    Route::resource('orders', AdminOrderController::class)->only(['index', 'show', 'destroy']);
    Route::put('/orders/{order}/status', [AdminOrderController::class, 'updateStatus'])->name('orders.updateStatus');
    Route::put('/orders/{order}/payment-status', [AdminOrderController::class, 'updatePaymentStatus'])->name('orders.updatePaymentStatus');
    Route::resource('users', UserController::class)->only(['index', 'show', 'destroy']);
    Route::post('/users/{user}/toggle-admin', [UserController::class, 'toggleAdmin'])->name('users.toggleAdmin');
    Route::put('/users/{user}/permissions', [UserController::class, 'updatePermissions'])->name('users.updatePermissions');
    Route::post('/users/{user}/impersonate', [UserController::class, 'impersonate'])->name('users.impersonate');
    Route::post('/users/stop-impersonate', [UserController::class, 'stopImpersonate'])->name('users.stopImpersonate');
    Route::get('/analytics', [AnalyticsController::class, 'index'])->name('analytics.index');
    Route::get('/analytics/active-users', [AnalyticsController::class, 'getActiveUsers'])->name('analytics.activeUsers');
    Route::get('/settings', [SettingsController::class, 'index'])->name('settings.index');
    Route::post('/settings', [SettingsController::class, 'update'])->name('settings.update');
    Route::post('/notifications/toggle', [\App\Http\Controllers\Admin\NotificationController::class, 'toggle'])->name('notifications.toggle');
    Route::get('/notifications/preferences', [\App\Http\Controllers\Admin\NotificationController::class, 'preferences'])->name('notifications.preferences');
    Route::get('/notifications/check', [\App\Http\Controllers\Admin\NotificationController::class, 'check'])->name('notifications.check');
    Route::get('/email-marketing', [\App\Http\Controllers\Admin\EmailMarketingController::class, 'index'])->name('email-marketing.index');
    Route::post('/email-marketing/send', [\App\Http\Controllers\Admin\EmailMarketingController::class, 'sendMarketingEmail'])->name('email-marketing.send');
    Route::post('/email-marketing/{id}/toggle', [\App\Http\Controllers\Admin\EmailMarketingController::class, 'toggleStatus'])->name('email-marketing.toggle');
    Route::delete('/email-marketing/{id}', [\App\Http\Controllers\Admin\EmailMarketingController::class, 'destroy'])->name('email-marketing.destroy');
    Route::post('/approve-store', [\App\Http\Controllers\Admin\ManualApprovalController::class, 'approveStore'])->name('manual-approval.approve-store');
    Route::post('/reject-store', [\App\Http\Controllers\Admin\ManualApprovalController::class, 'rejectStore'])->name('manual-approval.reject-store');
    Route::post('/suspend-store', [\App\Http\Controllers\Admin\ManualApprovalController::class, 'suspendStore'])->name('manual-approval.suspend-store');
    Route::post('/unsuspend-store', [\App\Http\Controllers\Admin\ManualApprovalController::class, 'unsuspendStore'])->name('manual-approval.unsuspend-store');
    
});

// Store Routes
Route::get('/store/register', [StoreController::class, 'showRegisterForm'])->name('store.register');
Route::post('/store/register', [StoreController::class, 'register'])->name('store.register.submit');


Route::middleware(['auth'])->prefix('store')->group(function () {
    Route::get('/dashboard', [App\Http\Controllers\Store\StoreController::class, 'dashboard'])->name('store.dashboard');
    Route::resource('products', App\Http\Controllers\Store\StoreProductController::class)->names('store.products');
    Route::get('request-quotes', [App\Http\Controllers\Store\RequestQuoteController::class, 'index'])->name('store.request-quotes.index');
    Route::post('request-quotes/{quote}/reply', [App\Http\Controllers\Store\RequestQuoteController::class, 'reply'])->name('store.request-quotes.reply');
    Route::resource('coupons', App\Http\Controllers\Store\CouponController::class)->names('store.coupons');
    Route::get('reports', [App\Http\Controllers\Store\ReportController::class, 'index'])->name('store.reports.index');
    Route::get('reviews', [App\Http\Controllers\Store\ReviewController::class, 'index'])->name('store.reviews.index');
    Route::get('withdraw', [App\Http\Controllers\Store\WithdrawController::class, 'index'])->name('store.withdraw.index');
    Route::post('withdraw/request', [App\Http\Controllers\Store\WithdrawController::class, 'request'])->name('store.withdraw.request');
    Route::get('returns', [App\Http\Controllers\Store\ReturnController::class, 'index'])->name('store.returns.index');
    Route::resource('staff', App\Http\Controllers\Store\StaffController::class)->names('store.staff');
    Route::get('followers', [App\Http\Controllers\Store\FollowerController::class, 'index'])->name('store.followers.index');
    Route::get('analytics', [App\Http\Controllers\Store\AnalyticsController::class, 'index'])->name('store.analytics.index');
    Route::get('announcements', [App\Http\Controllers\Store\AnnouncementController::class, 'index'])->name('store.announcements.index');
    Route::get('tools', [App\Http\Controllers\Store\ToolController::class, 'index'])->name('store.tools.index');
    Route::resource('orders', App\Http\Controllers\Store\OrderController::class)->only(['index'])->names('store.orders');
});

// Store profile (must be after all other /store routes to avoid route conflict)
Route::get('/store/{slug}', [StoreController::class, 'show'])->name('store.show');

// Alias: let /stores/* redirect to /store/* for user error tolerance
Route::any('/stores/{any?}', function($any = null) {
    $path = $any ? "/store/$any" : '/store/dashboard';
    return redirect($path);
})->where('any', '.*');



