@extends('layouts.app')

@section('title', 'Products')

@section('content')
<style>
    .product-image-container {
        position: relative;
    }
    .product-images-wrapper {
        position: relative;
        width: 100%;
        height: 100%;
        display: flex;
        transition: transform 0.5s ease-in-out;
    }
    .product-image {
        min-width: 100%;
        height: 100%;
        object-fit: cover;
        display: block;
    }
    .product-image-container:hover .product-images-wrapper {
        animation: slideImages 3s infinite;
    }
    @keyframes slideImages {
        0%, 20% {
            transform: translateX(0);
        }
        25%, 45% {
            transform: translateX(-100%);
        }
        50%, 70% {
            transform: translateX(-200%);
        }
        75%, 95% {
            transform: translateX(-300%);
        }
        100% {
            transform: translateX(0);
        }
    }
    .product-image-container[data-image-count="1"]:hover .product-images-wrapper {
        animation: none;
    }
    .product-image-container[data-image-count="2"]:hover .product-images-wrapper {
        animation: slideImages2 3s infinite;
    }
    @keyframes slideImages2 {
        0%, 45% {
            transform: translateX(0);
        }
        50%, 95% {
            transform: translateX(-100%);
        }
        100% {
            transform: translateX(0);
        }
    }
    .product-image-container[data-image-count="3"]:hover .product-images-wrapper {
        animation: slideImages3 4s infinite;
    }
    @keyframes slideImages3 {
        0%, 30% {
            transform: translateX(0);
        }
        33%, 63% {
            transform: translateX(-100%);
        }
        66%, 96% {
            transform: translateX(-200%);
        }
        100% {
            transform: translateX(0);
        }
    }
</style>
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
    <div class="flex flex-col md:flex-row gap-8">
        <!-- Sidebar Filters -->
        <aside class="w-full md:w-64 bg-white p-6 rounded-lg shadow-md h-fit">
            <h2 class="font-display text-2xl font-bold mb-6">Filters</h2>
            
            <form method="GET" action="{{ route('products.index') }}" class="space-y-6">
                <!-- Search -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Search</label>
                    <input type="text" name="search" value="{{ request('search') }}" placeholder="Search products..." class="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-primary">
                </div>

                <!-- Category -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Category</label>
                    <select name="category" class="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-primary">
                        <option value="">All Categories</option>
                        @foreach($categories as $category)
                            <option value="{{ $category->id }}" {{ request('category') == $category->id ? 'selected' : '' }}>
                                {{ $category->name }}
                            </option>
                        @endforeach
                    </select>
                </div>

                <!-- Price Range -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Price Range</label>
                    <div class="flex gap-2">
                        <input type="number" name="min_price" value="{{ request('min_price') }}" placeholder="Min" class="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-primary">
                        <input type="number" name="max_price" value="{{ request('max_price') }}" placeholder="Max" class="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-primary">
                    </div>
                </div>

                <!-- Sort -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Sort By</label>
                    <select name="sort" class="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-primary">
                        <option value="latest" {{ request('sort') == 'latest' ? 'selected' : '' }}>Latest</option>
                        <option value="price_low" {{ request('sort') == 'price_low' ? 'selected' : '' }}>Price: Low to High</option>
                        <option value="price_high" {{ request('sort') == 'price_high' ? 'selected' : '' }}>Price: High to Low</option>
                        <option value="name" {{ request('sort') == 'name' ? 'selected' : '' }}>Name A-Z</option>
                    </select>
                </div>

                <button type="submit" class="w-full bg-primary text-white px-4 py-2 rounded-md hover:bg-primary-dark transition">Apply Filters</button>
                <a href="{{ route('products.index') }}" class="block text-center text-gray-600 hover:text-primary transition">Clear Filters</a>
            </form>
        </aside>

        <!-- Products Grid -->
        <div class="flex-1">
            <div class="mb-6">
                <h1 class="font-display text-3xl font-bold text-gray-900">All Products</h1>
                <p class="text-gray-600 mt-2">{{ $products->total() }} products found</p>
            </div>

            @if($products->count() > 0)
                <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6">
                    @foreach($products as $product)
                        <div class="bg-white rounded-lg overflow-hidden">
                            <a href="{{ route('products.show', $product->slug) }}">
                                <div class="relative h-64 bg-gray-200 overflow-hidden product-image-container" data-image-count="{{ 1 + ($product->images && is_array($product->images) ? count($product->images) : 0) }}">
                                    <div class="product-images-wrapper">
                                        <img src="{{ productImageUrl($product->image) }}" alt="{{ $product->name }}" class="product-image w-full h-full object-cover"
                                             onerror="this.src='{{ url('images/placeholder.jpg') }}'; this.onerror=null;">
                                @if($product->images && is_array($product->images) && count($product->images) > 0)
                                    @foreach($product->images as $image)
                                                <img src="{{ productImageUrl($image) }}" alt="{{ $product->name }}" class="product-image w-full h-full object-cover"
                                                     onerror="this.src='{{ url('images/placeholder.jpg') }}'; this.onerror=null;">
                                            @endforeach
                                        @endif
                                    </div>
                                    @if($product->compare_price)
                                        <span class="absolute top-3 left-3 bg-green-500 text-white px-2 py-1 text-xs font-bold z-10">
                                            {{ round((($product->compare_price - $product->price) / $product->compare_price) * 100) }}% OFF
                                        </span>
                                    @endif
                                    <div class="absolute top-3 right-3 flex flex-col gap-2 z-10">
                                        <a href="{{ route('products.show', $product->slug) }}" class="w-8 h-8 bg-white flex items-center justify-center shadow-sm" onclick="event.stopPropagation();" title="View Product">
                                            <svg class="w-4 h-4 text-gray-700" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"/>
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"/>
                                            </svg>
                                        </a>
                                        @auth
                                            @php
                                                $isInWishlist = auth()->user()->wishlists()->where('product_id', $product->id)->exists();
                                            @endphp
                                            <form action="{{ $isInWishlist ? route('customer.wishlist.remove', $product->id) : route('customer.wishlist.add', $product->id) }}" method="POST" class="wishlist-form" onclick="event.stopPropagation(); event.preventDefault(); addToWishlist(this, {{ $product->id }});">
                                                @csrf
                                                @if($isInWishlist)
                                                    @method('DELETE')
                                                @endif
                                                <button type="submit" class="w-8 h-8 bg-white flex items-center justify-center shadow-sm" title="{{ $isInWishlist ? 'Remove from Wishlist' : 'Add to Wishlist' }}">
                                                    <svg class="w-4 h-4 text-gray-700" fill="{{ $isInWishlist ? 'currentColor' : 'none' }}" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z"/>
                                                    </svg>
                                                </button>
                                            </form>
                                        @else
                                            <a href="{{ route('login') }}" class="w-8 h-8 bg-white flex items-center justify-center shadow-sm" onclick="event.stopPropagation();" title="Login to Add to Wishlist">
                                                <svg class="w-4 h-4 text-gray-700" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z"/>
                                                </svg>
                                            </a>
                                        @endauth
                                        <form action="{{ route('cart.store') }}" method="POST" class="add-to-cart-form" onclick="event.stopPropagation();">
                                            @csrf
                                            <input type="hidden" name="product_id" value="{{ $product->id }}">
                                            <input type="hidden" name="quantity" value="1">
                                            <button type="submit" class="w-8 h-8 bg-white flex items-center justify-center shadow-sm" title="Add to Cart">
                                                <svg class="w-4 h-4 text-gray-700" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 3h2l.4 2M7 13h10l4-8H5.4M7 13L5.4 5M7 13l-2.293 2.293c-.63.63-.184 1.707.707 1.707H17m0 0a2 2 0 100 4 2 2 0 000-4zm-8 2a2 2 0 11-4 0 2 2 0 014 0z"/>
                                                </svg>
                                            </button>
                                        </form>
                                    </div>
                                </div>
                                <div class="bg-black px-4 py-3 flex items-center justify-between">
                                    <span class="text-white text-sm font-medium">{{ $product->name }}</span>
                                    <span class="text-white font-semibold text-base">{{ currency($product->price) }}</span>
                                </div>
                            </a>
                        </div>
                    @endforeach
                </div>

                <!-- Pagination -->
                <div class="mt-8">
                    {{ $products->links() }}
                </div>
            @else
                <div class="text-center py-12">
                    <p class="text-gray-600 text-lg">No products found. Try adjusting your filters.</p>
                </div>
            @endif
        </div>
    </div>
</div>

@push('scripts')
<script>
// Wishlist functionality
function addToWishlist(form, productId) {
    const formData = new FormData(form);
    const url = form.action;
    const method = form.querySelector('input[name="_method"]')?.value || 'POST';
    
    fetch(url, {
        method: method,
        body: formData,
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Update button state
            const button = form.querySelector('button');
            const svg = button.querySelector('svg');
            if (data.added) {
                button.classList.add('text-primary');
                svg.setAttribute('fill', 'currentColor');
                // Update form action for removal
                form.action = '/wishlist/' + productId;
                if (!form.querySelector('input[name="_method"]')) {
                    const methodInput = document.createElement('input');
                    methodInput.type = 'hidden';
                    methodInput.name = '_method';
                    methodInput.value = 'DELETE';
                    form.appendChild(methodInput);
                } else {
                    form.querySelector('input[name="_method"]').value = 'DELETE';
                }
            } else {
                button.classList.remove('text-primary');
                svg.setAttribute('fill', 'none');
                // Update form action for addition
                form.action = '/wishlist/' + productId;
                const methodInput = form.querySelector('input[name="_method"]');
                if (methodInput) {
                    methodInput.remove();
                }
            }
        }
    })
    .catch(error => {
        console.error('Wishlist error:', error);
    });
}
</script>
@endpush
@endsection

