@extends('layouts.app')

@section('title', 'My Addresses')

@section('content')
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
    <div class="mb-8 flex items-center justify-between">
        <div>
            <h1 class="text-3xl font-bold text-gray-900">My Addresses</h1>
            <p class="text-gray-600 mt-2">Manage your shipping and billing addresses</p>
        </div>
        <button onclick="openAddAddressModal()" 
                class="bg-primary text-white px-6 py-3 rounded-lg font-semibold hover:bg-primary-dark transition">
            + Add New Address
        </button>
    </div>

    @if(session('success'))
        <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 mb-6 rounded" role="alert">
            <p>{{ session('success') }}</p>
        </div>
    @endif

    @if($addresses->count() > 0)
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            @foreach($addresses as $address)
                <div class="bg-white rounded-lg shadow-md p-6 border-2 {{ $address->is_default ? 'border-primary' : 'border-gray-200' }}">
                    <div class="flex items-start justify-between mb-4">
                        <div>
                            <span class="inline-block px-3 py-1 text-xs font-semibold rounded-full {{ $address->is_default ? 'bg-primary text-white' : 'bg-gray-100 text-gray-800' }}">
                                {{ $address->is_default ? 'Default' : ucfirst($address->type) }}
                            </span>
                        </div>
                        <div class="flex items-center space-x-2">
                            <button onclick="openEditAddressModal({{ $address->id }})" 
                                    class="text-primary hover:text-primary-dark">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"/>
                                </svg>
                            </button>
                            <form action="{{ route('customer.addresses.delete', $address->id) }}" method="POST" class="inline">
                                @csrf
                                @method('DELETE')
                                <button type="submit" 
                                        class="text-red-600 hover:text-red-800"
                                        onclick="return confirm('Delete this address?')">
                                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/>
                                    </svg>
                                </button>
                            </form>
                        </div>
                    </div>
                    <div class="space-y-2">
                        <p class="font-semibold text-gray-900">{{ $address->full_name }}</p>
                        <p class="text-sm text-gray-600">{{ $address->phone }}</p>
                        <p class="text-sm text-gray-600">{{ $address->address }}</p>
                        @if($address->city || $address->state || $address->postal_code)
                            <p class="text-sm text-gray-600">
                                {{ $address->city }}{{ $address->city && $address->state ? ', ' : '' }}{{ $address->state }}
                                @if($address->postal_code) {{ $address->postal_code }} @endif
                            </p>
                        @endif
                        @if($address->country)
                            <p class="text-sm text-gray-600">{{ $address->country }}</p>
                        @endif
                    </div>
                    @if(!$address->is_default)
                        <form action="{{ route('customer.addresses.set-default', $address->id) }}" method="POST" class="mt-4">
                            @csrf
                            <button type="submit" 
                                    class="text-sm text-primary hover:text-primary-dark font-semibold">
                                Set as Default
                            </button>
                        </form>
                    @endif
                </div>
            @endforeach
        </div>
    @else
        <div class="bg-white rounded-lg shadow-md p-12 text-center">
            <svg class="w-24 h-24 text-gray-400 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z"/>
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 11a3 3 0 11-6 0 3 3 0 016 0z"/>
            </svg>
            <h3 class="text-xl font-bold text-gray-900 mb-2">No addresses saved</h3>
            <p class="text-gray-600 mb-6">Add your first address to make checkout faster!</p>
            <button onclick="openAddAddressModal()" 
                    class="inline-block bg-primary text-white px-6 py-3 rounded-lg font-semibold hover:bg-primary-dark transition">
                Add Address
            </button>
        </div>
    @endif
</div>

<!-- Add/Edit Address Modal -->
<div id="addressModal" class="hidden fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center p-4">
    <div class="bg-white rounded-lg shadow-xl max-w-md w-full p-6 max-h-[90vh] overflow-y-auto">
        <div class="flex items-center justify-between mb-6">
            <h2 id="modalTitle" class="text-2xl font-bold text-gray-900">Add New Address</h2>
            <button onclick="closeAddressModal()" class="text-gray-400 hover:text-gray-600">
                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                </svg>
            </button>
        </div>

        <form id="addressForm" method="POST" action="{{ route('customer.addresses.store') }}">
            @csrf
            <input type="hidden" name="_method" id="formMethod" value="POST">
            <input type="hidden" name="address_id" id="addressId">
            
            @if(request()->routeIs('customer.addresses.update'))
                @method('PUT')
            @endif

            <div class="space-y-4">
                <div>
                    <label for="type" class="block text-sm font-semibold text-gray-700 mb-2">Address Type</label>
                    <select name="type" id="type" required class="w-full px-4 py-3 border-2 border-gray-200 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent">
                        <option value="shipping">Shipping</option>
                        <option value="billing">Billing</option>
                    </select>
                </div>

                <div>
                    <label for="full_name" class="block text-sm font-semibold text-gray-700 mb-2">Full Name *</label>
                    <input type="text" name="full_name" id="full_name" required
                           class="w-full px-4 py-3 border-2 border-gray-200 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent">
                </div>

                <div>
                    <label for="phone" class="block text-sm font-semibold text-gray-700 mb-2">Phone *</label>
                    <input type="text" name="phone" id="phone" required
                           class="w-full px-4 py-3 border-2 border-gray-200 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent">
                </div>

                <div>
                    <label for="address" class="block text-sm font-semibold text-gray-700 mb-2">Address *</label>
                    <textarea name="address" id="address" rows="3" required
                              class="w-full px-4 py-3 border-2 border-gray-200 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent"></textarea>
                </div>

                <div class="grid grid-cols-2 gap-4">
                    <div>
                        <label for="city" class="block text-sm font-semibold text-gray-700 mb-2">City</label>
                        <input type="text" name="city" id="city"
                               class="w-full px-4 py-3 border-2 border-gray-200 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent">
                    </div>

                    <div>
                        <label for="state" class="block text-sm font-semibold text-gray-700 mb-2">State</label>
                        <input type="text" name="state" id="state"
                               class="w-full px-4 py-3 border-2 border-gray-200 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent">
                    </div>
                </div>

                <div class="grid grid-cols-2 gap-4">
                    <div>
                        <label for="postal_code" class="block text-sm font-semibold text-gray-700 mb-2">Postal Code</label>
                        <input type="text" name="postal_code" id="postal_code"
                               class="w-full px-4 py-3 border-2 border-gray-200 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent">
                    </div>

                    <div>
                        <label for="country" class="block text-sm font-semibold text-gray-700 mb-2">Country</label>
                        <input type="text" name="country" id="country"
                               class="w-full px-4 py-3 border-2 border-gray-200 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent">
                    </div>
                </div>

                <div class="flex items-center">
                    <input type="checkbox" name="is_default" id="is_default" value="1"
                           class="rounded border-gray-300 text-primary focus:ring-primary">
                    <label for="is_default" class="ml-2 text-sm text-gray-700">Set as default address</label>
                </div>
            </div>

            <div class="mt-6 flex items-center space-x-4">
                <button type="submit" 
                        class="flex-1 bg-primary text-white px-6 py-3 rounded-lg font-semibold hover:bg-primary-dark transition">
                    Save Address
                </button>
                <button type="button" onclick="closeAddressModal()" 
                        class="px-6 py-3 border-2 border-gray-200 rounded-lg font-semibold text-gray-700 hover:bg-gray-50 transition">
                    Cancel
                </button>
            </div>
        </form>
    </div>
</div>

<script>
function openAddAddressModal() {
    document.getElementById('modalTitle').textContent = 'Add New Address';
    document.getElementById('addressForm').action = '{{ route('customer.addresses.store') }}';
    document.getElementById('formMethod').value = 'POST';
    document.getElementById('addressForm').reset();
    document.getElementById('addressModal').classList.remove('hidden');
}

function openEditAddressModal(addressId) {
    // Fetch address data and populate form
    fetch(`{{ url('/addresses') }}/${addressId}/edit`, {
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json',
        }
    })
        .then(response => response.json())
        .then(data => {
            document.getElementById('modalTitle').textContent = 'Edit Address';
            document.getElementById('addressForm').action = `{{ url('/addresses') }}/${addressId}`;
            document.getElementById('formMethod').value = 'PUT';
            document.getElementById('addressId').value = addressId;
            document.getElementById('type').value = data.type;
            document.getElementById('full_name').value = data.full_name;
            document.getElementById('phone').value = data.phone;
            document.getElementById('address').value = data.address;
            document.getElementById('city').value = data.city || '';
            document.getElementById('state').value = data.state || '';
            document.getElementById('postal_code').value = data.postal_code || '';
            document.getElementById('country').value = data.country || '';
            document.getElementById('is_default').checked = data.is_default == 1;
            document.getElementById('addressModal').classList.remove('hidden');
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Failed to load address data');
        });
}

function closeAddressModal() {
    document.getElementById('addressModal').classList.add('hidden');
}

// Close modal when clicking outside
document.getElementById('addressModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeAddressModal();
    }
});
</script>
@endsection

