@extends('layouts.app')

@section('title', 'Analytics Dashboard')

@section('content')
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
    <div class="mb-8 flex items-center justify-between">
        <div>
            <h1 class="text-3xl font-bold text-gray-900">Analytics Dashboard</h1>
            <p class="text-gray-600 mt-2">Comprehensive user analytics and insights</p>
        </div>
        <div class="flex items-center gap-4">
            <select id="periodSelect" class="border border-gray-300 rounded-md px-4 py-2 focus:outline-none focus:ring-2 focus:ring-primary">
                <option value="1h" {{ $period == '1h' ? 'selected' : '' }}>Last Hour</option>
                <option value="5h" {{ $period == '5h' ? 'selected' : '' }}>Last 5 Hours</option>
                <option value="24h" {{ $period == '24h' || !in_array($period, ['1h', '5h', '24h', '3d', '7d', '2w', '1m']) ? 'selected' : '' }}>Last 24 Hours</option>
                <option value="3d" {{ $period == '3d' ? 'selected' : '' }}>Last 3 Days</option>
                <option value="7d" {{ $period == '7d' ? 'selected' : '' }}>Last 7 Days</option>
                <option value="2w" {{ $period == '2w' ? 'selected' : '' }}>Last 2 Weeks</option>
                <option value="1m" {{ $period == '1m' ? 'selected' : '' }}>Last 1 Month</option>
            </select>
        </div>
    </div>

    <!-- Key Metrics -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
        <div class="bg-gradient-to-br from-blue-500 to-blue-600 shadow-lg rounded-lg p-6 text-white">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-sm font-medium text-blue-100">Total Page Views</p>
                    <p class="text-3xl font-bold mt-2">{{ number_format($totalPageViews) }}</p>
                </div>
                <div class="bg-white/20 p-3 rounded-full">
                    <svg class="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"/>
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"/>
                    </svg>
                </div>
            </div>
        </div>

        <div class="bg-gradient-to-br from-green-500 to-green-600 shadow-lg rounded-lg p-6 text-white">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-sm font-medium text-green-100">Unique Visitors</p>
                    <p class="text-3xl font-bold mt-2">{{ number_format($uniqueVisitors) }}</p>
                </div>
                <div class="bg-white/20 p-3 rounded-full">
                    <svg class="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z"/>
                    </svg>
                </div>
            </div>
        </div>

        <div class="bg-gradient-to-br from-purple-500 to-purple-600 shadow-lg rounded-lg p-6 text-white">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-sm font-medium text-purple-100">Total Clicks</p>
                    <p class="text-3xl font-bold mt-2">{{ number_format($totalClicks) }}</p>
                </div>
                <div class="bg-white/20 p-3 rounded-full">
                    <svg class="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 15l-2 5L9 9l11 4-5 2zm0 0l5 5M7.188 2.239l.777 2.897M5.136 7.965l-2.898-.777M13.95 4.05l-2.122 2.122m-5.657 5.656l-2.12 2.122"/>
                    </svg>
                </div>
            </div>
        </div>

        <div class="bg-gradient-to-br from-orange-500 to-orange-600 shadow-lg rounded-lg p-6 text-white">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-sm font-medium text-orange-100">Active Users</p>
                    <p class="text-3xl font-bold mt-2" id="activeUsersCount">{{ $activeUsers }}</p>
                    <p class="text-xs text-orange-100 mt-1">Last 5 minutes</p>
                </div>
                <div class="bg-white/20 p-3 rounded-full">
                    <svg class="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 10V3L4 14h7v7l9-11h-7z"/>
                    </svg>
                </div>
            </div>
        </div>
    </div>

    <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
        <!-- Page Views Chart -->
        <div class="bg-white shadow rounded-lg p-6">
            <h2 class="text-xl font-bold text-gray-900 mb-4">Page Views</h2>
            <div class="h-64 flex items-end justify-between gap-1 overflow-x-auto">
                @foreach($pageViewsData as $data)
                    @php
                        $maxViews = count($pageViewsData) > 0 ? max(array_column($pageViewsData, 'views')) : 1;
                    @endphp
                    <div class="flex-1 flex flex-col items-center {{ count($pageViewsData) > 14 ? 'min-w-[50px]' : (count($pageViewsData) > 7 ? 'min-w-[40px]' : '') }}">
                        <div class="w-full bg-blue-200 rounded-t relative" style="height: {{ $data['views'] > 0 && $maxViews > 0 ? max(20, ($data['views'] / $maxViews) * 100) : 0 }}%">
                            <div class="absolute inset-x-0 bottom-full mb-1 text-xs text-gray-600 text-center whitespace-nowrap">
                                {{ $data['views'] }}
                            </div>
                        </div>
                        <p class="text-xs text-gray-500 mt-2 {{ count($pageViewsData) > 14 ? 'transform -rotate-90 origin-top-left whitespace-nowrap' : '' }}">{{ $data['date'] }}</p>
                    </div>
                @endforeach
            </div>
        </div>

        <!-- Activity Timeline -->
        <div class="bg-white shadow rounded-lg p-6">
            <h2 class="text-xl font-bold text-gray-900 mb-4">Activity Timeline</h2>
            <div class="h-64 flex items-end justify-between gap-1 overflow-x-auto">
                @foreach($activityTimeline as $activity)
                    <div class="flex-1 flex flex-col items-center {{ count($activityTimeline) > 24 ? 'min-w-[40px]' : 'min-w-[30px]' }}">
                        @php
                            $maxPageViews = count($activityTimeline) > 0 ? max(array_column($activityTimeline, 'page_views')) : 1;
                            $maxClicks = count($activityTimeline) > 0 ? max(array_column($activityTimeline, 'clicks')) : 1;
                        @endphp
                        <div class="w-full bg-purple-200 rounded-t mb-1" style="height: {{ $activity['page_views'] > 0 && $maxPageViews > 0 ? max(10, ($activity['page_views'] / $maxPageViews) * 100) : 0 }}%"></div>
                        <div class="w-full bg-orange-200 rounded-t" style="height: {{ $activity['clicks'] > 0 && $maxClicks > 0 ? max(10, ($activity['clicks'] / $maxClicks) * 100) : 0 }}%"></div>
                        <p class="text-xs text-gray-500 mt-2 {{ count($activityTimeline) > 24 ? 'transform -rotate-90 origin-top-left whitespace-nowrap' : 'transform -rotate-45 origin-top-left' }}">{{ $activity['hour'] }}</p>
                    </div>
                @endforeach
            </div>
            <div class="flex gap-4 mt-4 justify-center">
                <div class="flex items-center gap-2">
                    <div class="w-4 h-4 bg-purple-200 rounded"></div>
                    <span class="text-xs text-gray-600">Page Views</span>
                </div>
                <div class="flex items-center gap-2">
                    <div class="w-4 h-4 bg-orange-200 rounded"></div>
                    <span class="text-xs text-gray-600">Clicks</span>
                </div>
            </div>
        </div>
    </div>

    <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
        <!-- Top Pages -->
        <div class="bg-white shadow rounded-lg p-6">
            <h2 class="text-xl font-bold text-gray-900 mb-4">Top Pages</h2>
            <div class="space-y-3">
                @forelse($topPages as $page)
                    <div class="flex items-center justify-between pb-3 border-b border-gray-200 last:border-0 last:pb-0">
                        <div class="flex-1">
                            <p class="font-semibold text-gray-900">{{ $page->path }}</p>
                            <p class="text-sm text-gray-500">{{ number_format($page->views) }} views</p>
                        </div>
                    </div>
                @empty
                    <p class="text-gray-500 text-center py-4">No page views yet</p>
                @endforelse
            </div>
        </div>

        <!-- Traffic Sources -->
        <div class="bg-white shadow rounded-lg p-6">
            <h2 class="text-xl font-bold text-gray-900 mb-4">Traffic Sources</h2>
            <div class="space-y-3">
                @forelse($trafficSources as $source)
                    <div class="flex items-center justify-between pb-3 border-b border-gray-200 last:border-0 last:pb-0">
                        <div class="flex-1">
                            <p class="font-semibold text-gray-900">{{ ucfirst($source->source ?: 'Direct') }}</p>
                            <p class="text-sm text-gray-500">{{ number_format($source->total_visits) }} visits</p>
                        </div>
                    </div>
                @empty
                    <p class="text-gray-500 text-center py-4">No traffic sources yet</p>
                @endforelse
            </div>
        </div>
    </div>

    <div class="grid grid-cols-1 lg:grid-cols-3 gap-6 mb-8">
        <!-- Device Types -->
        <div class="bg-white shadow rounded-lg p-6">
            <h2 class="text-xl font-bold text-gray-900 mb-4">Device Types</h2>
            <div class="space-y-3">
                @forelse($deviceTypes as $device)
                    <div class="flex items-center justify-between">
                        <span class="text-gray-700 capitalize">{{ $device->device_type ?: 'Unknown' }}</span>
                        <span class="font-semibold text-gray-900">{{ number_format($device->count) }}</span>
                    </div>
                @empty
                    <p class="text-gray-500 text-center py-4">No data</p>
                @endforelse
            </div>
        </div>

        <!-- Browsers -->
        <div class="bg-white shadow rounded-lg p-6">
            <h2 class="text-xl font-bold text-gray-900 mb-4">Browsers</h2>
            <div class="space-y-3">
                @forelse($browsers as $browser)
                    <div class="flex items-center justify-between">
                        <span class="text-gray-700">{{ $browser->browser }}</span>
                        <span class="font-semibold text-gray-900">{{ number_format($browser->count) }}</span>
                    </div>
                @empty
                    <p class="text-gray-500 text-center py-4">No data</p>
                @endforelse
            </div>
        </div>

        <!-- Operating Systems -->
        <div class="bg-white shadow rounded-lg p-6">
            <h2 class="text-xl font-bold text-gray-900 mb-4">Operating Systems</h2>
            <div class="space-y-3">
                @forelse($operatingSystems as $os)
                    <div class="flex items-center justify-between">
                        <span class="text-gray-700">{{ $os->os }}</span>
                        <span class="font-semibold text-gray-900">{{ number_format($os->count) }}</span>
                    </div>
                @empty
                    <p class="text-gray-500 text-center py-4">No data</p>
                @endforelse
            </div>
        </div>
    </div>

    <!-- Active Users -->
    <div class="bg-white shadow rounded-lg p-6 mb-8">
        <div class="flex items-center justify-between mb-4">
            <h2 class="text-xl font-bold text-gray-900">Active Users (Last 5 Minutes)</h2>
            <button onclick="refreshActiveUsers()" class="text-primary hover:text-primary-dark text-sm font-medium">Refresh</button>
        </div>
        <div id="activeUsersList" class="space-y-3">
            @forelse($recentActiveUsers as $session)
                <div class="flex items-center justify-between pb-3 border-b border-gray-200 last:border-0 last:pb-0">
                    <div class="flex-1">
                        <p class="font-semibold text-gray-900">{{ $session->user ? $session->user->name : 'Guest' }}</p>
                        <p class="text-sm text-gray-500">{{ $session->landing_page }} • {{ $session->device_type }} • {{ $session->browser }}</p>
                        <p class="text-xs text-gray-400 mt-1">{{ $session->last_activity_at->diffForHumans() }}</p>
                    </div>
                </div>
            @empty
                <p class="text-gray-500 text-center py-4">No active users</p>
            @endforelse
        </div>
    </div>

    <!-- Top Referrers -->
    <div class="bg-white shadow rounded-lg p-6 mb-8">
        <h2 class="text-xl font-bold text-gray-900 mb-4">Top Referrers</h2>
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Domain</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Source</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Visits</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Page Views</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Last Seen</th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    @forelse($topReferrers as $referrer)
                        <tr>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">{{ $referrer->domain }}</td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-600">{{ ucfirst($referrer->source ?: 'Direct') }}</td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm font-semibold text-gray-900">{{ number_format($referrer->visits) }}</td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-600">{{ number_format($referrer->page_views) }}</td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">{{ $referrer->last_seen_at->diffForHumans() }}</td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="5" class="px-6 py-4 text-center text-gray-500">No referrers yet</td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
    </div>

    <!-- Top Clicked Links -->
    <div class="bg-white shadow rounded-lg p-6">
        <h2 class="text-xl font-bold text-gray-900 mb-4">Top Clicked Links</h2>
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Link Text</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">URL</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Clicks</th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    @forelse($topClicks as $click)
                        <tr>
                            <td class="px-6 py-4 text-sm text-gray-900">{{ Str::limit($click->element_text ?: 'N/A', 50) }}</td>
                            <td class="px-6 py-4 text-sm text-gray-600">{{ Str::limit($click->url, 60) }}</td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm font-semibold text-gray-900">{{ number_format($click->click_count) }}</td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="3" class="px-6 py-4 text-center text-gray-500">No clicks tracked yet</td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Period selector
    document.getElementById('periodSelect').addEventListener('change', function() {
        window.location.href = '{{ route("admin.analytics.index") }}?period=' + this.value;
    });

    // Auto-refresh active users every 30 seconds
    setInterval(refreshActiveUsers, 30000);
});

function refreshActiveUsers() {
    fetch('{{ route("admin.analytics.activeUsers") }}')
        .then(response => response.json())
        .then(data => {
            document.getElementById('activeUsersCount').textContent = data.count;
            
            let html = '';
            if (data.users.length > 0) {
                data.users.forEach(user => {
                    html += `
                        <div class="flex items-center justify-between pb-3 border-b border-gray-200 last:border-0 last:pb-0">
                            <div class="flex-1">
                                <p class="font-semibold text-gray-900">${user.user}</p>
                                <p class="text-sm text-gray-500">${user.page} • ${user.device} • ${user.browser}</p>
                                <p class="text-xs text-gray-400 mt-1">${user.last_activity}</p>
                            </div>
                        </div>
                    `;
                });
            } else {
                html = '<p class="text-gray-500 text-center py-4">No active users</p>';
            }
            document.getElementById('activeUsersList').innerHTML = html;
        })
        .catch(error => console.error('Error fetching active users:', error));
}
</script>

@push('scripts')
<script>
// Click tracking for all links and buttons
document.addEventListener('DOMContentLoaded', function() {
    document.addEventListener('click', function(e) {
        const target = e.target.closest('a, button, [onclick]');
        if (!target) return;
        
        const elementType = target.tagName.toLowerCase();
        const elementId = target.id || null;
        const elementClass = target.className || null;
        const elementText = target.textContent?.trim().substring(0, 100) || null;
        const url = target.href || target.getAttribute('data-url') || null;
        const pageUrl = window.location.pathname;

        // Send click data to server
        fetch('{{ route("analytics.trackClick") }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            },
            body: JSON.stringify({
                element_type: elementType,
                element_id: elementId,
                element_class: elementClass,
                element_text: elementText,
                url: url,
                page_url: pageUrl
            })
        }).catch(error => console.error('Click tracking error:', error));
    });
});
</script>
@endpush
@endsection

