<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Drop existing tables if they exist
        Schema::dropIfExists('referrers');
        Schema::dropIfExists('user_sessions');
        Schema::dropIfExists('clicks');
        Schema::dropIfExists('page_views');

        // Page Views Table
        Schema::create('page_views', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->nullable()->constrained()->onDelete('set null');
            $table->string('session_id')->index();
            $table->string('url');
            $table->string('path');
            $table->string('method')->default('GET');
            $table->string('ip_address', 45)->nullable();
            $table->text('user_agent')->nullable();
            $table->string('referrer')->nullable();
            $table->string('device_type')->nullable(); // desktop, mobile, tablet
            $table->string('browser')->nullable();
            $table->string('os')->nullable();
            $table->string('country')->nullable();
            $table->string('city')->nullable();
            $table->integer('duration')->nullable(); // Time spent on page in seconds
            $table->timestamp('viewed_at');
            $table->timestamps();
            
            $table->index(['session_id', 'viewed_at']);
            $table->index(['user_id', 'viewed_at']);
            $table->index('viewed_at');
        });

        // Clicks Table
        Schema::create('clicks', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->nullable()->constrained()->onDelete('set null');
            $table->string('session_id')->index();
            $table->string('element_type'); // link, button, image, etc.
            $table->string('element_id')->nullable();
            $table->string('element_class')->nullable();
            $table->text('element_text')->nullable();
            $table->string('url')->nullable();
            $table->string('page_url');
            $table->string('ip_address', 45)->nullable();
            $table->text('user_agent')->nullable();
            $table->timestamp('clicked_at');
            $table->timestamps();
            
            $table->index(['session_id', 'clicked_at']);
            $table->index(['user_id', 'clicked_at']);
            $table->index('clicked_at');
        });

        // User Sessions Table
        Schema::create('user_sessions', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->nullable()->constrained()->onDelete('set null');
            $table->string('session_id')->unique()->index();
            $table->string('ip_address', 45)->nullable();
            $table->text('user_agent')->nullable();
            $table->string('device_type')->nullable();
            $table->string('browser')->nullable();
            $table->string('os')->nullable();
            $table->string('country')->nullable();
            $table->string('city')->nullable();
            $table->string('referrer')->nullable();
            $table->string('landing_page');
            $table->integer('page_views')->default(0);
            $table->integer('clicks')->default(0);
            $table->integer('duration')->default(0); // Total session duration in seconds
            $table->timestamp('started_at');
            $table->timestamp('last_activity_at');
            $table->timestamp('ended_at')->nullable();
            $table->boolean('is_active')->default(true);
            $table->timestamps();
            
            $table->index(['user_id', 'last_activity_at']);
            $table->index(['is_active', 'last_activity_at']);
            $table->index('last_activity_at');
        });

        // Referrers Table
        Schema::create('referrers', function (Blueprint $table) {
            $table->id();
            $table->string('referrer_url');
            $table->string('domain');
            $table->string('source')->nullable(); // google, facebook, direct, etc.
            $table->string('medium')->nullable(); // organic, cpc, referral, etc.
            $table->string('campaign')->nullable();
            $table->integer('visits')->default(0);
            $table->integer('unique_visitors')->default(0);
            $table->integer('page_views')->default(0);
            $table->timestamp('first_seen_at');
            $table->timestamp('last_seen_at');
            $table->timestamps();
            
            $table->index('domain');
            $table->index('source');
            $table->index('last_seen_at');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('referrers');
        Schema::dropIfExists('user_sessions');
        Schema::dropIfExists('clicks');
        Schema::dropIfExists('page_views');
    }
};
