<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\URL;
use App\Models\Setting;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     *
     * @return void
     */
    public function register()
    {
        //
    }

    /**
     * Bootstrap any application services.
     *
     * @return void
     */
    public function boot()
    {
        // Fix for MySQL key length issue
        Schema::defaultStringLength(191);
        
        // Set base URL from .env file (APP_URL)
        // Force URL if APP_URL is set (works for both local subdirectory and production)
        $appUrl = config('app.url');
        
        // If APP_URL is set and contains a path (subdirectory), always use it
        // This ensures route() and url() helpers generate correct URLs with subdirectory
        if ($appUrl) {
            $parsedUrl = parse_url($appUrl);
            // If URL has a path component (subdirectory), force it
            if (isset($parsedUrl['path']) && $parsedUrl['path'] !== '/' && $parsedUrl['path'] !== '') {
                URL::forceRootUrl($appUrl);
                $scheme = $parsedUrl['scheme'] ?? 'http';
                URL::forceScheme($scheme);
            } elseif (config('app.env') !== 'local') {
                // In production without subdirectory, also force it
                URL::forceRootUrl($appUrl);
                $scheme = parse_url($appUrl, PHP_URL_SCHEME) ?? 'http';
                URL::forceScheme($scheme);
            }
        }
        
        // Set default currency to GHS if not set
        if (!session()->has('currency')) {
            session(['currency' => 'GHS']);
        }
        
        // Share currency service and settings with all views
        view()->composer('*', function ($view) {
            $currencyService = app(\App\Services\CurrencyService::class);
            $view->with('currencyService', $currencyService);
            $view->with('currentCurrency', $currencyService->getCurrentCurrency());
            $view->with('supportedCurrencies', $currencyService->getSupportedCurrencies());
            
            // Share settings if settings table exists
            if (Schema::hasTable('settings')) {
                $view->with('siteTitle', \App\Models\Setting::get('site_title', 'Vibrant Vogue'));
                $view->with('siteLogo', \App\Models\Setting::get('site_logo'));
                $view->with('siteFavicon', \App\Models\Setting::get('site_favicon'));
                $view->with('primaryColor', \App\Models\Setting::get('primary_color', '#ff5349'));
                $view->with('facebookUrl', \App\Models\Setting::get('facebook_url', ''));
                $view->with('whatsappUrl', \App\Models\Setting::get('whatsapp_url', ''));
                $view->with('instagramUrl', \App\Models\Setting::get('instagram_url', ''));
                $view->with('youtubeUrl', \App\Models\Setting::get('youtube_url', ''));
                $view->with('snapchatUrl', \App\Models\Setting::get('snapchat_url', ''));
                $view->with('tiktokUrl', \App\Models\Setting::get('tiktok_url', ''));
            } else {
                $view->with('siteTitle', 'Vibrant Vogue');
                $view->with('siteLogo', null);
                $view->with('siteFavicon', null);
                $view->with('primaryColor', '#ff5349');
                $view->with('facebookUrl', '');
                $view->with('whatsappUrl', '');
                $view->with('instagramUrl', '');
                $view->with('youtubeUrl', '');
                $view->with('snapchatUrl', '');
                $view->with('tiktokUrl', '');
            }
            
            // Share active categories for navigation
            if (Schema::hasTable('categories')) {
                $view->with('navCategories', \App\Models\Category::where('is_active', true)->orderBy('name')->get());
            } else {
                $view->with('navCategories', collect());
            }
        });
    }
}
