<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;
use App\Models\Withdrawal;
use App\Models\Review;
use App\Models\Announcement;
use App\Models\StoreStaff;
use App\Models\ReturnRequest;

class Store extends Model
{
    use HasFactory;

    protected $table = 'stores';

    protected $fillable = [
        'user_id',
        'business_name',
        'slug',
        'description',
        'phone',
        'logo',
        'cover_image',
        'address',
        'city',
        'state',
        'country',
        'postal_code',
        'website',
        'facebook_url',
        'instagram_url',
        'twitter_url',
        'commission_rate',
        'status',
        'rejection_reason',
        'approved_at',
        'total_sales',
        'total_earnings',
        'pending_earnings',
        'rating',
        'total_reviews',
        'is_featured',
    ];

    protected $casts = [
        'commission_rate' => 'decimal:2',
        'total_earnings' => 'decimal:2',
        'pending_earnings' => 'decimal:2',
        'approved_at' => 'datetime',
        'is_featured' => 'boolean',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($store) {
            if (empty($store->slug)) {
                $store->slug = Str::slug($store->business_name);
                
                // Ensure slug is unique
                $originalSlug = $store->slug;
                $counter = 1;
                while (static::where('slug', $store->slug)->exists()) {
                    $store->slug = $originalSlug . '-' . $counter;
                    $counter++;
                }
            }
        });
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function products()
    {
        return $this->hasMany(Product::class);
    }

    public function coupons()
    {
        return $this->hasMany(Coupon::class);
    }

    public function quotes()
    {
        return $this->hasMany(Quote::class);
    }

    public function orders()
    {
        return $this->hasManyThrough(Order::class, OrderItem::class, 'product_id', 'id', 'id', 'order_id')
            ->whereHas('items.product', function($query) {
                $query->where('store_id', $this->id);
            });
    }

    public function followers()
    {
        return $this->belongsToMany(User::class, 'store_followers');
    }

    public function announcements()
    {
        return $this->hasMany(Announcement::class);
    }

    public function returns()
    {
        return $this->hasMany(ReturnRequest::class);
    }

    public function staff()
    {
        return $this->hasMany(StoreStaff::class);
    }

    public function withdrawals()
    {
        return $this->hasMany(Withdrawal::class);
    }

    public function reviews()
    {
        return $this->hasMany(Review::class);
    }

    public function isApproved()
    {
        return $this->status === 'approved';
    }

    public function isPending()
    {
        return $this->status === 'pending';
    }

    public function isSuspended()
    {
        return $this->status === 'suspended';
    }
}

