<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use App\Models\PageView;
use App\Models\UserSession;
use App\Models\Referrer;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class TrackAnalytics
{
    /**
     * Handle an incoming request.
     */
    public function handle(Request $request, Closure $next)
    {
        // Skip tracking for admin routes and API routes
        if ($request->is('admin/*') || $request->is('api/*')) {
            return $next($request);
        }

        $response = $next($request);

        // Track analytics asynchronously to avoid slowing down the response
        try {
            $this->trackPageView($request);
        } catch (\Exception $e) {
            // Log error but don't break the request
            \Log::error('Analytics tracking error: ' . $e->getMessage());
        }

        return $response;
    }

    protected function trackPageView(Request $request)
    {
        $sessionId = $request->session()->getId();
        $userId = auth()->id();
        $url = $request->fullUrl();
        $path = $request->path();
        $referrer = $request->header('referer');
        $userAgent = $request->userAgent();
        $ipAddress = $request->ip();

        // Parse user agent
        $deviceInfo = $this->parseUserAgent($userAgent);

        // Get or create session
        $session = UserSession::firstOrCreate(
            ['session_id' => $sessionId],
            [
                'user_id' => $userId,
                'ip_address' => $ipAddress,
                'user_agent' => $userAgent,
                'device_type' => $deviceInfo['device_type'],
                'browser' => $deviceInfo['browser'],
                'os' => $deviceInfo['os'],
                'referrer' => $referrer,
                'landing_page' => $path,
                'started_at' => now(),
                'last_activity_at' => now(),
                'is_active' => true,
            ]
        );

        // Update session activity
        $session->update([
            'last_activity_at' => now(),
            'page_views' => $session->page_views + 1,
        ]);

        // Track page view
        PageView::create([
            'user_id' => $userId,
            'session_id' => $sessionId,
            'url' => $url,
            'path' => $path,
            'method' => $request->method(),
            'ip_address' => $ipAddress,
            'user_agent' => $userAgent,
            'referrer' => $referrer,
            'device_type' => $deviceInfo['device_type'],
            'browser' => $deviceInfo['browser'],
            'os' => $deviceInfo['os'],
            'viewed_at' => now(),
        ]);

        // Track referrer
        if ($referrer && !$request->is('admin/*')) {
            $this->trackReferrer($referrer);
        }
    }

    protected function trackReferrer($referrerUrl)
    {
        $parsed = parse_url($referrerUrl);
        $domain = $parsed['host'] ?? 'direct';
        
        if ($domain === 'direct' || empty($domain)) {
            $source = 'direct';
            $medium = 'none';
        } else {
            $source = $this->identifySource($domain);
            $medium = $this->identifyMedium($referrerUrl);
        }

        $referrer = Referrer::firstOrNew(['domain' => $domain]);
        
        if (!$referrer->exists) {
            $referrer->referrer_url = $referrerUrl;
            $referrer->source = $source;
            $referrer->medium = $medium;
            $referrer->first_seen_at = now();
            $referrer->visits = 0;
            $referrer->unique_visitors = 0;
            $referrer->page_views = 0;
        }

        $referrer->visits += 1;
        $referrer->page_views += 1;
        $referrer->last_seen_at = now();
        $referrer->save();
    }

    protected function identifySource($domain)
    {
        $domain = strtolower($domain);
        
        if (str_contains($domain, 'google')) return 'google';
        if (str_contains($domain, 'facebook')) return 'facebook';
        if (str_contains($domain, 'twitter')) return 'twitter';
        if (str_contains($domain, 'instagram')) return 'instagram';
        if (str_contains($domain, 'youtube')) return 'youtube';
        if (str_contains($domain, 'linkedin')) return 'linkedin';
        
        return 'other';
    }

    protected function identifyMedium($url)
    {
        $url = strtolower($url);
        
        if (str_contains($url, 'utm_medium=cpc') || str_contains($url, 'gclid')) return 'cpc';
        if (str_contains($url, 'utm_medium=organic')) return 'organic';
        if (str_contains($url, 'utm_medium=email')) return 'email';
        if (str_contains($url, 'utm_medium=social')) return 'social';
        
        return 'referral';
    }

    protected function parseUserAgent($userAgent)
    {
        $deviceType = 'desktop';
        $browser = 'unknown';
        $os = 'unknown';

        // Detect device type
        if (preg_match('/mobile|android|iphone|ipad|ipod|blackberry|iemobile|opera mini/i', $userAgent)) {
            $deviceType = 'mobile';
        } elseif (preg_match('/tablet|ipad/i', $userAgent)) {
            $deviceType = 'tablet';
        }

        // Detect browser
        if (preg_match('/chrome/i', $userAgent) && !preg_match('/edg/i', $userAgent)) {
            $browser = 'Chrome';
        } elseif (preg_match('/firefox/i', $userAgent)) {
            $browser = 'Firefox';
        } elseif (preg_match('/safari/i', $userAgent) && !preg_match('/chrome/i', $userAgent)) {
            $browser = 'Safari';
        } elseif (preg_match('/edg/i', $userAgent)) {
            $browser = 'Edge';
        } elseif (preg_match('/opera|opr/i', $userAgent)) {
            $browser = 'Opera';
        }

        // Detect OS
        if (preg_match('/windows/i', $userAgent)) {
            $os = 'Windows';
        } elseif (preg_match('/macintosh|mac os x/i', $userAgent)) {
            $os = 'macOS';
        } elseif (preg_match('/linux/i', $userAgent)) {
            $os = 'Linux';
        } elseif (preg_match('/android/i', $userAgent)) {
            $os = 'Android';
        } elseif (preg_match('/iphone|ipad|ipod/i', $userAgent)) {
            $os = 'iOS';
        }

        return [
            'device_type' => $deviceType,
            'browser' => $browser,
            'os' => $os,
        ];
    }
}
