<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Category;
use App\Services\CurrencyService;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    public function index(Request $request)
    {
        $query = Product::where('is_active', true)->with('category');

        // Search
        if ($request->has('search') && $request->search) {
            $searchTerm = $request->search;
            
            // Store search term in cookie for recommendations
            $searchHistory = json_decode($request->cookie('user_search_history', '[]'), true);
            if (!in_array($searchTerm, $searchHistory)) {
                $searchHistory[] = $searchTerm;
                // Keep only last 10 searches
                $searchHistory = array_slice($searchHistory, -10);
            }
            
            $query->where(function($q) use ($searchTerm) {
                $q->where('name', 'like', '%' . $searchTerm . '%')
                  ->orWhere('description', 'like', '%' . $searchTerm . '%')
                  ->orWhere('short_description', 'like', '%' . $searchTerm . '%')
                  ->orWhere('brand', 'like', '%' . $searchTerm . '%')
                  ->orWhere('sku', 'like', '%' . $searchTerm . '%');
            });
        }

        // Category filter
        if ($request->has('category') && $request->category) {
            $query->where('category_id', $request->category);
        }

        // Featured filter
        if ($request->has('featured') && $request->featured) {
            $query->where('is_featured', true);
        }

        // Discount filter
        if ($request->has('discount') && $request->discount) {
            $query->whereNotNull('compare_price')
                  ->whereColumn('compare_price', '>', 'price');
        }

        // Price filter
        if ($request->has('min_price') && $request->min_price) {
            $query->where('price', '>=', $request->min_price);
        }
        if ($request->has('max_price') && $request->max_price) {
            $query->where('price', '<=', $request->max_price);
        }

        // Sort
        $sort = $request->get('sort', 'latest');
        switch ($sort) {
            case 'price_low':
                $query->orderBy('price', 'asc');
                break;
            case 'price_high':
                $query->orderBy('price', 'desc');
                break;
            case 'name':
                $query->orderBy('name', 'asc');
                break;
            default:
                $query->latest();
        }

        $products = $query->paginate(12);
        $categories = Category::where('is_active', true)->get();

        $response = response()->view('products.index', compact('products', 'categories'));
        
        // Set search history cookie if search was performed
        if ($request->has('search') && $request->search) {
            $searchHistory = json_decode($request->cookie('user_search_history', '[]'), true);
            if (!in_array($request->search, $searchHistory)) {
                $searchHistory[] = $request->search;
                $searchHistory = array_slice($searchHistory, -10);
            }
            $response->cookie('user_search_history', json_encode($searchHistory), 60 * 24 * 30); // 30 days
        }
        
        return $response;
    }

    public function show($slug, Request $request)
    {
        $product = Product::where('slug', $slug)
            ->where('is_active', true)
            ->with('category')
            ->firstOrFail();

        // Increment views
        $product->increment('views');

        // Store viewed product info in cookies for recommendations
        $viewedCategories = json_decode($request->cookie('user_viewed_categories', '[]'), true);
        if ($product->category_id && !in_array($product->category_id, $viewedCategories)) {
            $viewedCategories[] = $product->category_id;
            $viewedCategories = array_slice($viewedCategories, -10); // Keep last 10 categories
        }
        
        $viewedProductIds = json_decode($request->cookie('user_viewed_products', '[]'), true);
        if (!in_array($product->id, $viewedProductIds)) {
            $viewedProductIds[] = $product->id;
            $viewedProductIds = array_slice($viewedProductIds, -20); // Keep last 20 products
        }

        // Related products
        $relatedProducts = Product::where('category_id', $product->category_id)
            ->where('id', '!=', $product->id)
            ->where('is_active', true)
            ->take(4)
            ->get();

        $response = response()->view('products.show', compact('product', 'relatedProducts'));
        
        // Set cookies
        $response->cookie('user_viewed_categories', json_encode($viewedCategories), 60 * 24 * 30); // 30 days
        $response->cookie('user_viewed_products', json_encode($viewedProductIds), 60 * 24 * 30); // 30 days
        
        return $response;
    }

    public function search(Request $request)
    {
        $query = $request->get('q', '');
        
        if (strlen($query) < 2) {
            return response()->json(['products' => []]);
        }

        $currencyService = app(CurrencyService::class);
        $currentCurrency = session('currency', 'GHS');

        $products = Product::where('is_active', true)
            ->where(function($q) use ($query) {
                $q->where('name', 'like', '%' . $query . '%')
                  ->orWhere('description', 'like', '%' . $query . '%')
                  ->orWhere('short_description', 'like', '%' . $query . '%')
                  ->orWhere('brand', 'like', '%' . $query . '%')
                  ->orWhere('sku', 'like', '%' . $query . '%');
            })
            ->with('category')
            ->take(8)
            ->get()
            ->map(function($product) use ($currencyService, $currentCurrency) {
                return [
                    'id' => $product->id,
                    'name' => $product->name,
                    'slug' => $product->slug,
                    'price' => $product->price,
                    'formatted_price' => $currencyService->format($product->price, $currentCurrency),
                    'image' => productImageUrl($product->image),
                    'category' => $product->category->name ?? 'N/A',
                ];
            });

        return response()->json(['products' => $products]);
    }
}
