<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Http;
use App\Models\Order;
use Illuminate\Support\Facades\Log;

class PaymentController extends Controller
{
    /**
     * Obtain Paystack payment information
     * @return void
     */
    public function handleGatewayCallback(Request $request)
    {
        $reference = $request->query('reference');

        if (!$reference) {
            return redirect()->route('checkout')->with('error', 'Payment failed: No reference provided.');
        }

        try {
            // Manual Verification to handle SSL issues
            $response = Http::withOptions(['verify' => false])
                ->withToken(config('paystack.secretKey'))
                ->get("https://api.paystack.co/transaction/verify/{$reference}");

            if ($response->successful()) {
                $paymentDetails = $response->json();

                if ($paymentDetails['status'] && $paymentDetails['data']['status'] === 'success') {
                    $orderId = $paymentDetails['data']['metadata']['order_id'] ?? null;
                    
                    if ($orderId) {
                        $order = Order::find($orderId);
                        
                        if ($order) {
                            $order->update([
                                'payment_status' => 'paid',
                                'status' => 'processing'
                            ]);
                            
                            return redirect()->route('orders.show', $order->id)->with('success', 'Payment successful!');
                        }
                    }
                }
            }

            Log::error('Paystack Verification Failed: ' . $response->body());
            return redirect()->route('checkout')->with('error', 'Payment verification failed.');

        } catch (\Exception $e) {
            Log::error('Paystack Verify Exception: ' . $e->getMessage());
            return redirect()->route('checkout')->with('error', 'Payment error: ' . $e->getMessage());
        }
    }
}
