<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\OrderItem;
use App\Models\CartItem;
use App\Events\NewOrderCreated;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class OrderController extends Controller
{
    public function index()
    {
        // Only authenticated users can view order history
        if (!Auth::check()) {
            return redirect()->route('login')->with('error', 'Please login to view your orders.');
        }

        $orders = Order::where('user_id', Auth::id())
            ->latest()
            ->paginate(10);

        return view('orders.index', compact('orders'));
    }

    public function show($id)
    {
        $order = Order::findOrFail($id);

        // Check if user owns this order or if it's a guest order
        if (Auth::check()) {
            // Authenticated user - check if they own this order
            if ($order->user_id && $order->user_id !== Auth::id()) {
                abort(403, 'Unauthorized access.');
            }
        } else {
            // Guest user - check if this order ID matches the one stored in their session or cookie
            // This allows guests to view their order confirmation immediately after placing it
            $guestOrderId = session('guest_order_id') ?? request()->cookie('guest_order_id');
            
            // Also check if order was just created (within last 10 minutes) as a fallback
            // This handles cases where session might not persist immediately after redirect
            $orderAge = now()->diffInMinutes($order->created_at);
            $isRecentOrder = $orderAge <= 10;
            
            if ($guestOrderId != $order->id && !$isRecentOrder) {
                // Order ID doesn't match session/cookie AND order is not recent
                return redirect()->route('home')->with('info', 'Please login to view order details, or check your email for order confirmation.');
            }
            // Allow if: session/cookie matches OR order is very recent (within 10 minutes)
        }

        $order->load('items.product');
        $primaryColor = \App\Models\Setting::get('primary_color', '#ff5349');

        return view('orders.show', compact('order', 'primaryColor'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'shipping_name' => 'required|string|max:255',
            'shipping_email' => 'required|email|max:255',
            'shipping_phone' => 'required|string|max:20',
            'shipping_address' => 'required|string',
            'payment_method' => 'required|string',
            'notes' => 'nullable|string',
        ]);

        // Get cart items
        if (Auth::check()) {
            $cartItems = CartItem::where('user_id', Auth::id())
                ->with('product')
                ->get()
                ->filter(function ($item) {
                    return $item->product !== null;
                })
                ->map(function ($item) {
                    $item->subtotal = $item->product->price * $item->quantity;
                    return $item;
                });
        } else {
            // Guest user - use session
            $cartItems = collect();
            $sessionCart = session('cart', []);
            
            foreach ($sessionCart as $item) {
                $product = \App\Models\Product::find($item['product_id']);
                if ($product) {
                    $cartItems->push((object) [
                        'product_id' => $product->id,
                        'product' => $product,
                        'quantity' => $item['quantity'],
                        'size' => $item['size'] ?? null,
                        'color' => $item['color'] ?? null,
                        'subtotal' => $product->price * $item['quantity'],
                    ]);
                }
            }
        }

        if ($cartItems->isEmpty()) {
            return redirect()->route('cart.index')->with('error', 'Your cart is empty!');
        }

        $subtotal = $cartItems->sum(function ($item) {
            return $item->subtotal;
        });

        $tax = 0; // No tax
        $shipping = 0; // No shipping
        $total = $subtotal;

        $order = Order::create([
            'user_id' => Auth::id(), // null for guests
            'order_number' => 'ORD-' . strtoupper(Str::random(10)),
            'subtotal' => $subtotal,
            'tax' => $tax,
            'shipping' => $shipping,
            'total' => $total,
            'status' => 'pending',
            'payment_status' => 'pending',
            'payment_method' => $request->payment_method ?? 'cash_on_delivery',
            'shipping_name' => $request->shipping_name,
            'shipping_email' => $request->shipping_email,
            'shipping_phone' => $request->shipping_phone,
            'shipping_address' => $request->shipping_address,
            'shipping_city' => '',
            'shipping_state' => null,
            'shipping_postal_code' => '',
            'shipping_country' => '',
            'notes' => $request->notes,
        ]);

        foreach ($cartItems as $cartItem) {
            OrderItem::create([
                'order_id' => $order->id,
                'product_id' => $cartItem->product_id,
                'product_name' => $cartItem->product->name,
                'product_image' => $cartItem->product->image,
                'price' => $cartItem->product->price,
                'quantity' => $cartItem->quantity,
                'size' => $cartItem->size,
                'color' => $cartItem->color,
                'subtotal' => $cartItem->subtotal,
            ]);
        }

        // Store order ID in session AND cookie for guests BEFORE clearing cart
        // This ensures guests can view their order confirmation immediately after placing it
        // Using both session and cookie as backup in case session doesn't persist immediately
        if (!Auth::check()) {
            session()->put('guest_order_id', $order->id);
            // Also set a cookie as backup (expires in 1 hour)
            cookie()->queue('guest_order_id', $order->id, 60);
            session()->save(); // Explicitly save session before redirect
        }

        // Clear cart AFTER storing order ID
        if (Auth::check()) {
            CartItem::where('user_id', Auth::id())->delete();
        } else {
            session()->forget('cart');
            session()->save(); // Save session after clearing cart
        }

        // Send order confirmation email
        try {
            $siteTitle = \App\Models\Setting::get('site_title', 'Vibrant Vogue');
            \Illuminate\Support\Facades\Mail::to($order->shipping_email)->send(
                new \App\Mail\OrderConfirmationEmail($order, $siteTitle)
            );
        } catch (\Exception $e) {
            \Log::error('Failed to send order confirmation email: ' . $e->getMessage());
            // Don't fail the order if email fails
        }

        // Fire event for admin notifications
        event(new NewOrderCreated($order));

        if ($request->payment_method === 'paystack') {
            try {
                // Debug logging
                $secretKey = config('paystack.secretKey');
                if (empty($secretKey)) {
                    Log::error('Paystack Error: Secret Key is MISSING in config');
                    return redirect()->route('orders.show', $order->id)
                        ->with('error', 'Payment configuration error. Please contact support.');
                }

                // Manual Paystack Initialization to handle SSL issues
                $response = Http::withOptions(['verify' => false]) // Bypass SSL verification for local/hosting issues
                    ->withToken($secretKey)
                    ->post('https://api.paystack.co/transaction/initialize', [
                        'email' => $order->shipping_email,
                        'amount' => $order->total * 100, // Amount in kobo
                        'currency' => 'GHS',
                        'reference' => 'PAY-' . Str::random(10) . '-' . time(),
                        'metadata' => [
                            'order_id' => $order->id,
                        ],
                        'callback_url' => route('payment.callback'),
                    ]);

                if ($response->successful()) {
                    $data = $response->json();
                    if (isset($data['status']) && $data['status'] && isset($data['data']['authorization_url'])) {
                        return redirect()->away($data['data']['authorization_url']);
                    }
                }
                
                Log::error('Paystack Init Failed: ' . $response->body());
                return redirect()->route('orders.show', $order->id)
                    ->with('error', 'Payment initialization failed. Please try again or choose another payment method.');

            } catch (\Exception $e) {
                Log::error('Paystack Exception: ' . $e->getMessage());
                return redirect()->route('orders.show', $order->id)
                    ->with('error', 'Payment initialization failed: ' . $e->getMessage());
            }
        }

        // Use redirect()->away() or ensure session is saved
        return redirect()->route('orders.show', $order->id)
            ->with('success', 'Order placed successfully! Your order number is: ' . $order->order_number);
    }

    public function checkout(Request $request)
    {
        // Checkout page ONLY displays existing cart items
        // NO automatic product additions - products must be added via "Add to Cart" or "Buy Now" buttons first
        
        // If any product_id parameters are present, remove them (prevent accidental additions from URL)
        if ($request->has('product_id') || $request->has('buy_now_token')) {
            return redirect()->route('checkout');
        }
        
        // Get cart items
        if (Auth::check()) {
            $cartItems = CartItem::where('user_id', Auth::id())
                ->with('product')
                ->get()
                ->filter(function ($item) {
                    return $item->product !== null;
                })
                ->map(function ($item) {
                    $item->subtotal = $item->product->price * $item->quantity;
                    return $item;
                });
        } else {
            // Guest user - use session
            $cartItems = collect();
            $sessionCart = session('cart', []);
            
            foreach ($sessionCart as $item) {
                $product = \App\Models\Product::find($item['product_id']);
                if ($product) {
                    $cartItems->push((object) [
                        'product_id' => $product->id,
                        'product' => $product,
                        'quantity' => $item['quantity'],
                        'size' => $item['size'] ?? null,
                        'color' => $item['color'] ?? null,
                        'subtotal' => $product->price * $item['quantity'],
                    ]);
                }
            }
        }

        if ($cartItems->isEmpty()) {
            return redirect()->route('cart.index')->with('error', 'Your cart is empty!');
        }

        $subtotal = $cartItems->sum(function ($item) {
            return $item->subtotal;
        });

        $tax = 0; // No tax
        $shipping = 0; // No shipping
        $total = $subtotal;

        // Clear all buy_now flags after successfully loading checkout page
        // This ensures flags don't persist and cause issues on subsequent visits
        $sessionData = session()->all();
        foreach ($sessionData as $key => $value) {
            if (strpos($key, 'buy_now_') === 0) {
                session()->forget($key);
            }
        }

        return view('orders.checkout', compact('cartItems', 'subtotal', 'tax', 'shipping', 'total'));
    }
}
