<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Category;
use App\Models\Slider;
use App\Models\Setting;
use Illuminate\Http\Request;

class HomeController extends Controller
{
    public function index(Request $request)
    {
        // Featured products for hero/trending sections
        $featuredProducts = Product::where('is_featured', true)
            ->where('is_active', true)
            ->with('category')
            ->latest()
            ->take(8)
            ->get();

        // Trending products - personalized based on cookies
        $trendingProducts = $this->getPersonalizedTrendingProducts($request);

        // Top picks (featured products)
        $topPicks = Product::where('is_featured', true)
            ->where('is_active', true)
            ->with('category')
            ->latest()
            ->take(6)
            ->get();

        // Latest products (for latest products section)
        $latestProducts = Product::where('is_active', true)
            ->with('category')
            ->latest()
            ->take(8)
            ->get();

        // Categories for navigation
        $categories = Category::where('is_active', true)
            ->withCount('products')
            ->orderBy('name')
            ->get();

        // Top 10 categories for homepage grid (filling 12 slots with 2 custom items - 6 column layout)
        $topCategories = Category::where('is_active', true)
            ->withCount('products')
            ->orderBy('products_count', 'desc')
            ->take(10)
            ->get();

        // Products for Her (you can filter by category name containing "women" or specific categories)
        $herProducts = Product::where('is_active', true)
            ->whereHas('category', function($query) {
                $query->where('name', 'like', '%women%')
                      ->orWhere('name', 'like', '%woman%')
                      ->orWhere('name', 'like', '%ladies%');
            })
            ->with('category')
            ->latest()
            ->take(4)
            ->get();

        // Products for Him (you can filter by category name containing "men" or specific categories)
        $himProducts = Product::where('is_active', true)
            ->whereHas('category', function($query) {
                $query->where('name', 'like', '%men%')
                      ->orWhere('name', 'like', '%man%')
                      ->orWhere('name', 'like', '%gentlemen%');
            })
            ->with('category')
            ->latest()
            ->take(4)
            ->get();

        // Hero banner sliders
        $sliders = Slider::where('is_active', true)
            ->orderBy('order')
            ->get();

        // Get banner settings and convert image paths to URLs
        $bannerSettings = [
            'featured' => [
                'banner_image' => productImageUrl(Setting::get('featured_banner_image', 'https://images.unsplash.com/photo-1542291026-7eec264c27ff?w=800&h=1200&fit=crop')),
                'banner_title' => Setting::get('featured_banner_title', 'Premium Fashion'),
                'banner_subtitle' => Setting::get('featured_banner_subtitle', '⭐ FEATURED COLLECTION'),
                'banner_description' => Setting::get('featured_banner_description', 'Discover our handpicked selection of the finest fashion items, carefully curated for style and quality.'),
                'banner_link' => Setting::get('featured_banner_link', route('products.index', ['featured' => 1])),
            ],
            'shop_for_her' => [
                'banner_image' => productImageUrl(Setting::get('shop_for_her_banner_image', 'https://images.unsplash.com/photo-1490481651871-ab68de25d43d?w=1200&h=1600&fit=crop')),
                'banner_title' => Setting::get('shop_for_her_banner_title', 'SHOP FOR HER'),
                'banner_subtitle' => Setting::get('shop_for_her_banner_subtitle', "Women's Collection"),
                'banner_description' => Setting::get('shop_for_her_banner_description', 'Explore our elegant collection of women\'s fashion, crafted for style and sophistication.'),
                'banner_link' => Setting::get('shop_for_her_banner_link', route('products.index', ['category' => $categories->where('name', 'like', '%women%')->first()->id ?? ''])),
            ],
            'shop_for_him' => [
                'banner_image' => productImageUrl(Setting::get('shop_for_him_banner_image', 'https://images.unsplash.com/photo-1617137968427-85924c800a22?w=1200&h=1600&fit=crop')),
                'banner_title' => Setting::get('shop_for_him_banner_title', 'SHOP FOR HIM'),
                'banner_subtitle' => Setting::get('shop_for_him_banner_subtitle', "Men's Collection"),
                'banner_description' => Setting::get('shop_for_him_banner_description', 'Discover our curated collection of men\'s fashion, designed for style and confidence.'),
                'banner_link' => Setting::get('shop_for_him_banner_link', route('products.index', ['category' => $categories->where('name', 'like', '%men%')->first()->id ?? ''])),
            ],
            'trending' => [
                'banner_image' => productImageUrl(Setting::get('trending_banner_image', 'https://images.unsplash.com/photo-1583743814966-8936f5b7be1a?w=800&h=1200&fit=crop')),
                'banner_title' => Setting::get('trending_banner_title', 'TRENDING NOW'),
                'banner_subtitle' => Setting::get('trending_banner_subtitle', 'Hot Collection'),
                'banner_description' => Setting::get('trending_banner_description', 'Explore our curated selection of the most popular fashion pieces that everyone is wearing right now.'),
                'banner_link' => Setting::get('trending_banner_link', route('products.index', ['trending' => 1])),
            ],
        ];

        // Get homepage settings
        $showFeaturedProductsSection = Setting::get('show_featured_products_section', '1') == '1';
        $showNewArrivalsSection = Setting::get('show_new_arrivals_section', '1') == '1';
        
        // Get banner active status
        $bannerActiveStatus = [
            'featured' => Setting::get('featured_banner_active', '1') == '1',
            'trending' => Setting::get('trending_banner_active', '1') == '1',
            'shop_for_her' => Setting::get('shop_for_her_banner_active', '1') == '1',
            'shop_for_him' => Setting::get('shop_for_him_banner_active', '1') == '1',
        ];

        // Top 8 most popular category sections each with popular products
        $topCategorySections = $topCategories->take(8)->map(function($category) {
            $products = $category->products()
                ->where('is_active', true)
                ->orderByDesc('views')
                ->latest()
                ->take(6)
                ->get();
            return (object)[
                'category' => $category,
                'products' => $products
            ];
        });

        return view('home', compact(
            'featuredProducts', 
            'trendingProducts', 
            'topPicks',
            'latestProducts', 
            'categories',
            'topCategories',
            'herProducts',
            'himProducts',
            'sliders',
            'bannerSettings',
            'showFeaturedProductsSection',
            'showNewArrivalsSection',
            'bannerActiveStatus',
            'topCategorySections'
        ));
    }

    /**
     * Get personalized trending products based on user cookies (search history, viewed products, etc.)
     */
    private function getPersonalizedTrendingProducts(Request $request)
    {
        $query = Product::where('is_active', true)->with('category');
        
        // Get user preferences from cookies
        $searchHistory = json_decode($request->cookie('user_search_history', '[]'), true);
        $viewedCategories = json_decode($request->cookie('user_viewed_categories', '[]'), true);
        $viewedProductIds = json_decode($request->cookie('user_viewed_products', '[]'), true);
        
        $hasPreferences = !empty($searchHistory) || !empty($viewedCategories) || !empty($viewedProductIds);
        
        if ($hasPreferences) {
            // Build query based on user preferences
            $query->where(function($q) use ($searchHistory, $viewedCategories, $viewedProductIds) {
                // Match by search terms
                if (!empty($searchHistory)) {
                    $q->where(function($searchQuery) use ($searchHistory) {
                        foreach ($searchHistory as $term) {
                            $searchQuery->orWhere('name', 'like', '%' . $term . '%')
                                       ->orWhere('description', 'like', '%' . $term . '%')
                                       ->orWhere('short_description', 'like', '%' . $term . '%')
                                       ->orWhere('brand', 'like', '%' . $term . '%');
                        }
                    });
                }
                
                // Match by viewed categories
                if (!empty($viewedCategories)) {
                    $q->orWhereIn('category_id', $viewedCategories);
                }
                
                // Exclude already viewed products
                if (!empty($viewedProductIds)) {
                    $q->whereNotIn('id', $viewedProductIds);
                }
            });
            
            // Order by relevance (views, then latest)
            $query->orderBy('views', 'desc')->orderBy('created_at', 'desc');
        } else {
            // No preferences - show most viewed products
            $query->orderBy('views', 'desc');
        }
        
        // Get 6 products
        $products = $query->take(6)->get();
        
        // If we don't have enough products, fill with most viewed
        if ($products->count() < 6) {
            $remaining = 6 - $products->count();
            $excludeIds = $products->pluck('id')->toArray();
            if (!empty($viewedProductIds)) {
                $excludeIds = array_merge($excludeIds, $viewedProductIds);
            }
            
            $additionalProducts = Product::where('is_active', true)
                ->whereNotIn('id', $excludeIds)
                ->with('category')
                ->orderBy('views', 'desc')
                ->take($remaining)
                ->get();
            
            $products = $products->merge($additionalProducts);
        }
        
        return $products->take(6);
    }
}
