<?php

namespace App\Http\Controllers;

use App\Models\CartItem;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CartController extends Controller
{
    public function index()
    {
        if (Auth::check()) {
            // Authenticated user - use database
            // Clean up orphaned cart items first
            CartItem::where('user_id', Auth::id())
                ->whereDoesntHave('product')
                ->delete();
            
            $cartItems = CartItem::where('user_id', Auth::id())
                ->with('product')
                ->get()
                ->filter(function ($item) {
                    return $item->product !== null;
                })
                ->map(function ($item) {
                    $item->subtotal = $item->product->price * $item->quantity;
                    return $item;
                });
        } else {
            // Guest user - use session
            $cartItems = collect();
            $sessionCart = session('cart', []);
            $cleanedCart = [];
            
            foreach ($sessionCart as $key => $item) {
                $product = Product::find($item['product_id']);
                if ($product) {
                    $cleanedCart[$key] = $item;
                    $cartItems->push((object) [
                        'id' => $key,
                        'product_id' => $product->id,
                        'product' => $product,
                        'quantity' => $item['quantity'],
                        'size' => $item['size'] ?? null,
                        'color' => $item['color'] ?? null,
                        'subtotal' => $product->price * $item['quantity'],
                    ]);
                }
            }
            
            // Update session with cleaned cart
            if (count($cleanedCart) !== count($sessionCart)) {
                session(['cart' => $cleanedCart]);
            }
        }

        $total = $cartItems->sum(function ($item) {
            return $item->subtotal;
        });

        return view('cart.index', compact('cartItems', 'total'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|integer|min:1',
            'size' => 'nullable|string',
            'color' => 'nullable|string',
        ]);

        $product = Product::findOrFail($request->product_id);
        $isBuyNow = $request->has('buy_now') && $request->buy_now == '1';

        if (Auth::check()) {
            // Authenticated user - use database
            if ($isBuyNow) {
                // Buy Now: Clear entire cart first, then add the product
                CartItem::where('user_id', Auth::id())->delete();
                
                CartItem::create([
                    'user_id' => Auth::id(),
                    'product_id' => $request->product_id,
                    'quantity' => $request->quantity,
                    'size' => $request->size,
                    'color' => $request->color,
                ]);
            } else {
                // Add to Cart: Normal behavior - check if exists and increment or create
                $cartItem = CartItem::where('user_id', Auth::id())
                    ->where('product_id', $request->product_id)
                    ->where('size', $request->size)
                    ->where('color', $request->color)
                    ->first();

                if ($cartItem) {
                    $cartItem->quantity += $request->quantity;
                    $cartItem->save();
                } else {
                    CartItem::create([
                        'user_id' => Auth::id(),
                        'product_id' => $request->product_id,
                        'quantity' => $request->quantity,
                        'size' => $request->size,
                        'color' => $request->color,
                    ]);
                }
            }
            $cartCount = CartItem::where('user_id', Auth::id())->whereHas('product')->count();
        } else {
            // Guest user - use session
            if ($isBuyNow) {
                // Buy Now: Clear entire cart first, then add the product
                $sessionCart = [];
            } else {
                // Add to Cart: Normal behavior
                $sessionCart = session('cart', []);
            }
            
            $key = md5($request->product_id . ($request->size ?? '') . ($request->color ?? ''));
            $sessionCart[$key] = [
                'product_id' => $request->product_id,
                'quantity' => $request->quantity,
                'size' => $request->size,
                'color' => $request->color,
            ];
            
            // Clean up orphaned items
            foreach ($sessionCart as $k => $item) {
                if (!Product::find($item['product_id'] ?? null)) {
                    unset($sessionCart[$k]);
                }
            }
            
            session(['cart' => $sessionCart]);
            $cartCount = count($sessionCart);
        }

        // Return JSON response for AJAX requests
        if ($request->ajax() || $request->wantsJson() || $request->header('X-Requested-With') === 'XMLHttpRequest') {
            return response()->json([
                'success' => true,
                'message' => 'Product added to cart!',
                'cart_count' => $cartCount,
                'product_name' => $product->name,
            ]);
        }

        // If "buy_now" parameter is present, redirect to checkout instead of cart
        if ($request->has('buy_now') && $request->buy_now == '1') {
            return redirect()->route('checkout')->with('success', 'Product added to cart!');
        }

        return redirect()->route('cart.index')->with('success', 'Product added to cart!');
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'quantity' => 'required|integer|min:1',
        ]);

        if (Auth::check()) {
            // Authenticated user - use database
            $cartItem = CartItem::where('id', $id)
                ->where('user_id', Auth::id())
                ->firstOrFail();

            $cartItem->update([
                'quantity' => $request->quantity,
            ]);
        } else {
            // Guest user - use session
            $sessionCart = session('cart', []);
            if (isset($sessionCart[$id])) {
                $sessionCart[$id]['quantity'] = $request->quantity;
                session(['cart' => $sessionCart]);
            } else {
                return redirect()->route('cart.index')->with('error', 'Item not found!');
            }
        }

        return redirect()->route('cart.index')->with('success', 'Cart updated!');
    }

    public function destroy($id)
    {
        if (Auth::check()) {
            // Authenticated user - use database
            $cartItem = CartItem::where('id', $id)
                ->where('user_id', Auth::id())
                ->firstOrFail();

            $cartItem->delete();
        } else {
            // Guest user - use session
            $sessionCart = session('cart', []);
            if (isset($sessionCart[$id])) {
                unset($sessionCart[$id]);
                session(['cart' => $sessionCart]);
            } else {
                return redirect()->route('cart.index')->with('error', 'Item not found!');
            }
        }

        return redirect()->route('cart.index')->with('success', 'Item removed from cart!');
    }
}
