<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = User::withCount(['orders', 'cartItems']);

        // Search by name or email
        if ($request->has('search') && $request->search !== '') {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        // Filter by admin status
        if ($request->has('is_admin') && $request->is_admin !== '') {
            $query->where('is_admin', $request->is_admin === '1');
        }

        $users = $query->latest()->paginate(20);

        return view('admin.users.index', compact('users'));
    }

    /**
     * Display the specified resource.
     */
    public function show(User $user)
    {
        $user->load(['orders' => function($query) {
            $query->latest()->limit(10);
        }]);
        
        $totalSpent = $user->orders()->sum('total');
        $totalOrders = $user->orders()->count();

        return view('admin.users.show', compact('user', 'totalSpent', 'totalOrders'));
    }

    /**
     * Toggle admin status.
     */
    public function toggleAdmin(User $user)
    {
        // Prevent removing admin status from yourself
        if ($user->id === auth()->id()) {
            return redirect()->back()
                ->with('error', 'You cannot change your own admin status!');
        }

        $isAdmin = !$user->is_admin;
        
        $updateData = ['is_admin' => $isAdmin];
        
        // If granting admin access, set default permissions
        if ($isAdmin) {
            $updateData = array_merge($updateData, [
                'can_manage_products' => true,
                'can_manage_orders' => true,
                'can_manage_users' => false, // Require explicit permission
                'can_manage_categories' => true,
                'can_manage_sliders' => true,
                'can_view_analytics' => true,
                'can_manage_settings' => false, // Require explicit permission
            ]);
        } else {
            // If revoking admin access, clear all permissions
            $updateData = array_merge($updateData, [
                'can_manage_products' => false,
                'can_manage_orders' => false,
                'can_manage_users' => false,
                'can_manage_categories' => false,
                'can_manage_sliders' => false,
                'can_view_analytics' => false,
                'can_manage_settings' => false,
            ]);
        }

        $user->update($updateData);

        $status = $user->is_admin ? 'granted' : 'revoked';
        return redirect()->back()
            ->with('success', "Admin access {$status} successfully!");
    }

    /**
     * Update user permissions.
     */
    public function updatePermissions(Request $request, User $user)
    {
        // Prevent changing permissions for yourself
        if ($user->id === auth()->id()) {
            return redirect()->back()
                ->with('error', 'You cannot change your own permissions!');
        }

        $validated = $request->validate([
            'can_manage_products' => 'nullable|boolean',
            'can_manage_orders' => 'nullable|boolean',
            'can_manage_users' => 'nullable|boolean',
            'can_manage_categories' => 'nullable|boolean',
            'can_manage_sliders' => 'nullable|boolean',
            'can_view_analytics' => 'nullable|boolean',
            'can_manage_settings' => 'nullable|boolean',
        ]);

        $user->update([
            'can_manage_products' => $request->has('can_manage_products') ? true : false,
            'can_manage_orders' => $request->has('can_manage_orders') ? true : false,
            'can_manage_users' => $request->has('can_manage_users') ? true : false,
            'can_manage_categories' => $request->has('can_manage_categories') ? true : false,
            'can_manage_sliders' => $request->has('can_manage_sliders') ? true : false,
            'can_view_analytics' => $request->has('can_view_analytics') ? true : false,
            'can_manage_settings' => $request->has('can_manage_settings') ? true : false,
        ]);

        return redirect()->back()
            ->with('success', 'User permissions updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user)
    {
        // Prevent deleting yourself
        if ($user->id === auth()->id()) {
            return redirect()->back()
                ->with('error', 'You cannot delete your own account!');
        }

        try {
            // Delete related data (orders, cart items, wishlists, etc.)
            // These should cascade delete based on database relationships, but we'll be explicit
            
            // Delete user's cart items
            $user->cartItems()->delete();
            
            // Delete user's wishlist items
            $user->wishlists()->delete();
            
            // Delete user's addresses
            $user->addresses()->delete();
            
            // Note: Orders are kept for historical records, but user_id will be set to null
            // If you want to delete orders too, uncomment the line below:
            // $user->orders()->delete();
            
            // Delete the user
            $user->delete();

            return redirect()->route('admin.users.index')
                ->with('success', 'User deleted successfully!');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to delete user: ' . $e->getMessage());
        }
    }

}

