<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Slider;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class SliderController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $sliders = Slider::orderBy('order')->get();
        
        // Get banner settings and convert image paths to URLs
        $bannerSettings = [
            'featured' => [
                'banner_image' => productImageUrl(Setting::get('featured_banner_image')),
                'banner_image_path' => Setting::get('featured_banner_image'), // Store original path for form
                'banner_title' => Setting::get('featured_banner_title', 'Premium Fashion'),
                'banner_subtitle' => Setting::get('featured_banner_subtitle', '⭐ FEATURED COLLECTION'),
                'banner_description' => Setting::get('featured_banner_description', 'Discover our handpicked selection of the finest fashion items, carefully curated for style and quality.'),
                'banner_link' => Setting::get('featured_banner_link'),
            ],
            'shop_for_her' => [
                'banner_image' => productImageUrl(Setting::get('shop_for_her_banner_image', 'https://images.unsplash.com/photo-1490481651871-ab68de25d43d?w=1200&h=1600&fit=crop')),
                'banner_image_path' => Setting::get('shop_for_her_banner_image', 'https://images.unsplash.com/photo-1490481651871-ab68de25d43d?w=1200&h=1600&fit=crop'),
                'banner_title' => Setting::get('shop_for_her_banner_title', 'SHOP FOR HER'),
                'banner_subtitle' => Setting::get('shop_for_her_banner_subtitle', "Women's Collection"),
                'banner_description' => Setting::get('shop_for_her_banner_description', 'Explore our elegant collection of women\'s fashion, crafted for style and sophistication.'),
                'banner_link' => Setting::get('shop_for_her_banner_link'),
            ],
            'shop_for_him' => [
                'banner_image' => productImageUrl(Setting::get('shop_for_him_banner_image', 'https://images.unsplash.com/photo-1617137968427-85924c800a22?w=1200&h=1600&fit=crop')),
                'banner_image_path' => Setting::get('shop_for_him_banner_image', 'https://images.unsplash.com/photo-1617137968427-85924c800a22?w=1200&h=1600&fit=crop'),
                'banner_title' => Setting::get('shop_for_him_banner_title', 'SHOP FOR HIM'),
                'banner_subtitle' => Setting::get('shop_for_him_banner_subtitle', "Men's Collection"),
                'banner_description' => Setting::get('shop_for_him_banner_description', 'Discover our curated collection of men\'s fashion, designed for style and confidence.'),
                'banner_link' => Setting::get('shop_for_him_banner_link'),
            ],
            'trending' => [
                'banner_image' => productImageUrl(Setting::get('trending_banner_image', 'https://images.unsplash.com/photo-1583743814966-8936f5b7be1a?w=800&h=1200&fit=crop')),
                'banner_image_path' => Setting::get('trending_banner_image', 'https://images.unsplash.com/photo-1583743814966-8936f5b7be1a?w=800&h=1200&fit=crop'),
                'banner_title' => Setting::get('trending_banner_title', 'TRENDING NOW'),
                'banner_subtitle' => Setting::get('trending_banner_subtitle', 'Hot Collection'),
                'banner_description' => Setting::get('trending_banner_description', 'Explore our curated selection of the most popular fashion pieces that everyone is wearing right now.'),
                'banner_link' => Setting::get('trending_banner_link', route('products.index', ['trending' => 1])),
            ],
        ];
        
        return view('admin.sliders.index', compact('sliders', 'bannerSettings'));
    }
    
    /**
     * Update banner settings
     */
    public function updateBannerSettings(Request $request)
    {
        $bannerType = $request->input('banner_type', 'featured');
        
        $validated = $request->validate([
            'banner_type' => 'required|string|in:featured,shop_for_her,shop_for_him,trending',
            'banner_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:10240',
            'banner_title' => 'nullable|string|max:255',
            'banner_subtitle' => 'nullable|string|max:255',
            'banner_description' => 'nullable|string',
            'banner_link' => 'nullable|string|max:255',
        ]);
        
        // Determine which banner settings to update based on banner_type
        $prefix = $bannerType . '_banner_';
        
        // Handle banner image upload
        if ($request->hasFile('banner_image')) {
            // Delete old banner image if exists
            $oldBannerImage = Setting::get($prefix . 'image');
            if ($oldBannerImage && (strpos($oldBannerImage, '/storage/') === 0 || strpos($oldBannerImage, '/public/storage/') === 0 || strpos($oldBannerImage, 'storage/') === 0)) {
                $oldImagePath = str_replace(['/storage/', '/public/storage/', 'storage/'], '', $oldBannerImage);
                Storage::disk('public')->delete($oldImagePath);
            }
            
            $imagePath = $request->file('banner_image')->store('banners', 'public');
            // Store relative path - will be converted to URL when displayed
            Setting::set($prefix . 'image', $imagePath);
        }
        
        // Update other banner settings
        if (isset($validated['banner_title'])) {
            Setting::set($prefix . 'title', $validated['banner_title']);
        }
        if (isset($validated['banner_subtitle'])) {
            Setting::set($prefix . 'subtitle', $validated['banner_subtitle']);
        }
        if (isset($validated['banner_description'])) {
            Setting::set($prefix . 'description', $validated['banner_description']);
        }
        if (isset($validated['banner_link'])) {
            Setting::set($prefix . 'link', $validated['banner_link'] ?? '');
        }
        
        return redirect()->route('admin.sliders.index')
            ->with('success', 'Banner settings updated successfully!');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.sliders.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Check if request is empty (PostTooLargeException occurred)
        // When PostTooLargeException happens, $_POST and $_FILES are empty
        if (empty($_POST) && empty($_FILES) && $request->method() === 'POST') {
            return back()->withErrors([
                'error' => 'The uploaded file is too large. Maximum file size is 10MB. Please reduce the image size and try again.'
            ])->withInput();
        }
        
        try {
            $validated = $request->validate([
                'price' => 'nullable|numeric|min:0',
                'title' => 'nullable|string|max:255',
                'subtitle' => 'nullable|string|max:255',
                'description' => 'nullable|string',
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:10240',
                'link' => 'nullable|string|max:255',
                'order' => 'nullable|integer|min:0',
                'is_active' => 'nullable|boolean',
            ]);

            $dataToSave = [
                'price' => $request->input('price') !== null && $request->input('price') !== '' ? (float)$request->input('price') : 0,
                'title' => $request->input('title') ?? '',
                'subtitle' => $request->input('subtitle') ?? '',
                'description' => $request->input('description') ?? '',
                'link' => $request->input('link'),
                'is_active' => $request->has('is_active') ? true : false,
                'order' => $request->input('order') ?? (Slider::max('order') ?? 0) + 1,
            ];

            // Handle image upload
            if ($request->hasFile('image')) {
                $imagePath = $request->file('image')->store('sliders', 'public');
                $dataToSave['image'] = $imagePath;
            } else {
                $dataToSave['image'] = '';
            }

            $slider = Slider::create($dataToSave);

            return redirect()->route('admin.sliders.index')
                ->with('success', 'Slider created successfully!');
        } catch (\Exception $e) {
            \Log::error('Slider creation error: ' . $e->getMessage());
            return back()->withErrors(['error' => 'An error occurred: ' . $e->getMessage()])->withInput();
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Slider $slider)
    {
        return view('admin.sliders.edit', compact('slider'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Slider $slider)
    {
        try {
            $validated = $request->validate([
                'price' => 'nullable|numeric|min:0',
                'title' => 'nullable|string|max:255',
                'subtitle' => 'nullable|string|max:255',
                'description' => 'nullable|string',
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:10240',
                'link' => 'nullable|string|max:255',
                'order' => 'nullable|integer|min:0',
                'is_active' => 'nullable|boolean',
            ]);

            $dataToUpdate = [];

            // Handle image upload
            if ($request->hasFile('image')) {
                // Delete old image if it exists and is stored locally
                if ($slider->image && (strpos($slider->image, '/storage/') === 0 || strpos($slider->image, '/public/storage/') === 0 || strpos($slider->image, 'storage/') === 0)) {
                    $oldImagePath = str_replace(['/storage/', '/public/storage/', 'storage/'], '', $slider->image);
                    Storage::disk('public')->delete($oldImagePath);
                }
                
                $imagePath = $request->file('image')->store('sliders', 'public');
                $dataToUpdate['image'] = $imagePath;
            }

            // Update all fields
            $dataToUpdate['price'] = $request->input('price') !== null && $request->input('price') !== '' ? (float)$request->input('price') : 0;
            $dataToUpdate['title'] = $request->input('title') ?? '';
            $dataToUpdate['subtitle'] = $request->input('subtitle') ?? '';
            $dataToUpdate['description'] = $request->input('description') ?? '';
            $dataToUpdate['link'] = $request->input('link');
            $dataToUpdate['is_active'] = $request->has('is_active') ? true : false;
            
            if ($request->has('order') && $request->input('order') !== null) {
                $dataToUpdate['order'] = (int)$request->input('order');
            }

            $slider->update($dataToUpdate);

            return redirect()->route('admin.sliders.index')
                ->with('success', 'Slider updated successfully!');
        } catch (\Exception $e) {
            \Log::error('Slider update error: ' . $e->getMessage());
            return back()->withErrors(['error' => 'An error occurred: ' . $e->getMessage()])->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Slider $slider)
    {
        // Delete image if it exists and is stored locally
        if ($slider->image && (strpos($slider->image, '/storage/') === 0 || strpos($slider->image, '/public/storage/') === 0 || strpos($slider->image, 'storage/') === 0)) {
            $imagePath = str_replace(['/storage/', '/public/storage/', 'storage/'], '', $slider->image);
            Storage::disk('public')->delete($imagePath);
        }

        $slider->delete();

        return redirect()->route('admin.sliders.index')
            ->with('success', 'Slider deleted successfully!');
    }
}
