<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class SettingsController extends Controller
{
    /**
     * Display the settings page.
     */
    public function index()
    {
        $settings = [
            'site_title' => Setting::get('site_title', 'Vibrant Vogue'),
            'site_logo' => Setting::get('site_logo'),
            'site_favicon' => Setting::get('site_favicon'),
            'primary_color' => Setting::get('primary_color', '#ff5349'),
            'facebook_url' => Setting::get('facebook_url', ''),
            'whatsapp_url' => Setting::get('whatsapp_url', ''),
            'instagram_url' => Setting::get('instagram_url', ''),
            'youtube_url' => Setting::get('youtube_url', ''),
            'snapchat_url' => Setting::get('snapchat_url', ''),
            'tiktok_url' => Setting::get('tiktok_url', ''),
            'show_featured_products_section' => Setting::get('show_featured_products_section', '1'),
            'show_new_arrivals_section' => Setting::get('show_new_arrivals_section', '1'),
            'featured_banner_active' => Setting::get('featured_banner_active', '1'),
            'trending_banner_active' => Setting::get('trending_banner_active', '1'),
            'shop_for_her_banner_active' => Setting::get('shop_for_her_banner_active', '1'),
            'shop_for_him_banner_active' => Setting::get('shop_for_him_banner_active', '1'),
        ];

        return view('admin.settings.index', compact('settings'));
    }

    /**
     * Update the settings.
     */
    public function update(Request $request)
    {
        $validated = $request->validate([
            'site_title' => 'required|string|max:255',
            'site_logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp,svg|max:5120',
            'site_favicon' => 'nullable|image|mimes:ico,png,jpg,jpeg|max:1024',
            'primary_color' => 'required|string|regex:/^#[0-9A-Fa-f]{6}$/',
            'facebook_url' => 'nullable|url|max:255',
            'whatsapp_url' => 'nullable|url|max:255',
            'instagram_url' => 'nullable|url|max:255',
            'youtube_url' => 'nullable|url|max:255',
            'snapchat_url' => 'nullable|url|max:255',
            'tiktok_url' => 'nullable|url|max:255',
            'show_featured_products_section' => 'nullable|in:0,1',
            'show_new_arrivals_section' => 'nullable|in:0,1',
            'featured_banner_active' => 'nullable|in:0,1',
            'trending_banner_active' => 'nullable|in:0,1',
            'shop_for_her_banner_active' => 'nullable|in:0,1',
            'shop_for_him_banner_active' => 'nullable|in:0,1',
        ]);

        // Update site title
        Setting::set('site_title', $validated['site_title']);
        
        // Update primary color
        Setting::set('primary_color', $validated['primary_color']);
        
        // Update social media URLs
        Setting::set('facebook_url', $validated['facebook_url'] ?? '');
        Setting::set('whatsapp_url', $validated['whatsapp_url'] ?? '');
        Setting::set('instagram_url', $validated['instagram_url'] ?? '');
        Setting::set('youtube_url', $validated['youtube_url'] ?? '');
        Setting::set('snapchat_url', $validated['snapchat_url'] ?? '');
        Setting::set('tiktok_url', $validated['tiktok_url'] ?? '');
        
        // Update homepage settings
        // Hidden input sends '0', checkbox sends '1' when checked
        // If checkbox is checked, both values may be sent, but we check if '1' is present
        $showFeaturedProductsValue = $request->input('show_featured_products_section', '0');
        if (is_array($showFeaturedProductsValue)) {
            $showFeaturedProducts = in_array('1', $showFeaturedProductsValue) ? '1' : '0';
        } else {
            $showFeaturedProducts = ($showFeaturedProductsValue == '1') ? '1' : '0';
        }
        Setting::set('show_featured_products_section', $showFeaturedProducts);
        
        $showNewArrivalsValue = $request->input('show_new_arrivals_section', '0');
        if (is_array($showNewArrivalsValue)) {
            $showNewArrivals = in_array('1', $showNewArrivalsValue) ? '1' : '0';
        } else {
            $showNewArrivals = ($showNewArrivalsValue == '1') ? '1' : '0';
        }
        Setting::set('show_new_arrivals_section', $showNewArrivals);
        
        // Update banner active status
        $bannerSettings = [
            'featured_banner_active',
            'trending_banner_active',
            'shop_for_her_banner_active',
            'shop_for_him_banner_active',
        ];
        
        foreach ($bannerSettings as $bannerSetting) {
            $bannerValue = $request->input($bannerSetting, '0');
            if (is_array($bannerValue)) {
                $bannerActive = in_array('1', $bannerValue) ? '1' : '0';
            } else {
                $bannerActive = ($bannerValue == '1') ? '1' : '0';
            }
            Setting::set($bannerSetting, $bannerActive);
        }

        // Handle logo upload
        if ($request->hasFile('site_logo')) {
            // Delete old logo if exists
            $oldLogo = Setting::get('site_logo');
            if ($oldLogo && (strpos($oldLogo, '/storage/') === 0 || strpos($oldLogo, '/public/storage/') === 0)) {
                $oldLogoPath = str_replace(['/storage/', '/public/storage/'], '', $oldLogo);
                Storage::disk('public')->delete($oldLogoPath);
            }

            $logoPath = $request->file('site_logo')->store('settings', 'public');
            // Store relative path - will be converted to URL when displayed
            Setting::set('site_logo', $logoPath);
        }

        // Handle favicon upload
        if ($request->hasFile('site_favicon')) {
            // Delete old favicon if exists
            $oldFavicon = Setting::get('site_favicon');
            if ($oldFavicon && (strpos($oldFavicon, '/storage/') === 0 || strpos($oldFavicon, '/public/storage/') === 0)) {
                $oldFaviconPath = str_replace(['/storage/', '/public/storage/'], '', $oldFavicon);
                Storage::disk('public')->delete($oldFaviconPath);
            }

            $faviconPath = $request->file('site_favicon')->store('settings', 'public');
            // Store relative path - will be converted to URL when displayed
            Setting::set('site_favicon', $faviconPath);
        }

        return redirect()->route('admin.settings.index')
            ->with('success', 'Settings updated successfully!');
    }
}
