<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $products = Product::with(['category', 'categories'])->latest()->paginate(20);
        return view('admin.products.index', compact('products'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $categories = Category::where('is_active', true)->orderBy('name')->get();
        return view('admin.products.create', compact('categories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'categories' => 'required|array|min:1',
            'categories.*' => 'exists:categories,id',
            'name' => 'required|string|max:255',
            'description' => 'required|string',
            'short_description' => 'nullable|string|max:500',
            'price' => 'required|numeric|min:0',
            'compare_price' => 'nullable|numeric|min:0',
            'sku' => 'nullable|string|max:255|unique:products,sku',
            'image' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:5120',
            'images' => 'nullable|array',
            'images.*' => 'image|mimes:jpeg,png,jpg,gif,webp|max:5120',
            'size' => 'nullable|string|max:255',
            'color' => 'nullable|string|max:255',
            'brand' => 'nullable|string|max:255',
            'stock' => 'required|integer|min:0',
            'is_featured' => 'boolean',
            'is_active' => 'boolean',
            'on_sale' => 'boolean',
        ]);

        // Generate slug from name
        $validated['slug'] = Str::slug($validated['name']);
        
        // Ensure slug is unique
        $originalSlug = $validated['slug'];
        $counter = 1;
        while (Product::where('slug', $validated['slug'])->exists()) {
            $validated['slug'] = $originalSlug . '-' . $counter;
            $counter++;
        }

        // Handle main image upload
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('products', 'public');
            // Store relative path - Storage::url() will handle URL generation
            $validated['image'] = $imagePath;
        }

        // Handle additional images upload
        if ($request->hasFile('images')) {
            $imagePaths = [];
            foreach ($request->file('images') as $image) {
                $path = $image->store('products', 'public');
                $imagePaths[] = $path;
            }
            $validated['images'] = $imagePaths;
        }

        $validated['is_featured'] = $request->has('is_featured');
        $validated['is_active'] = $request->has('is_active');
        $validated['on_sale'] = $request->has('on_sale');
        
        // Set category_id to first category for backward compatibility
        $validated['category_id'] = $validated['categories'][0] ?? null;
        
        // Remove categories from validated array before creating product
        $categories = $validated['categories'];
        unset($validated['categories']);

        $product = Product::create($validated);
        
        // Sync categories
        $product->categories()->sync($categories);

        return redirect()->route('admin.products.index')
            ->with('success', 'Product created successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(Product $product)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Product $product)
    {
        $categories = Category::where('is_active', true)->orderBy('name')->get();
        $product->load('categories');
        return view('admin.products.edit', compact('product', 'categories'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Product $product)
    {
        $validated = $request->validate([
            'categories' => 'required|array|min:1',
            'categories.*' => 'exists:categories,id',
            'name' => 'required|string|max:255',
            'description' => 'required|string',
            'short_description' => 'nullable|string|max:500',
            'price' => 'required|numeric|min:0',
            'compare_price' => 'nullable|numeric|min:0',
            'sku' => 'nullable|string|max:255|unique:products,sku,' . $product->id,
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:5120',
            'images' => 'nullable|array',
            'images.*' => 'image|mimes:jpeg,png,jpg,gif,webp|max:5120',
            'size' => 'nullable|string|max:255',
            'color' => 'nullable|string|max:255',
            'brand' => 'nullable|string|max:255',
            'stock' => 'required|integer|min:0',
            'is_featured' => 'boolean',
            'is_active' => 'boolean',
            'on_sale' => 'boolean',
        ]);

        // Update slug if name changed
        if ($validated['name'] !== $product->name) {
            $validated['slug'] = Str::slug($validated['name']);
            $originalSlug = $validated['slug'];
            $counter = 1;
            while (Product::where('slug', $validated['slug'])->where('id', '!=', $product->id)->exists()) {
                $validated['slug'] = $originalSlug . '-' . $counter;
                $counter++;
            }
        }

        // Handle main image upload
        if ($request->hasFile('image')) {
            // Delete old image
            if ($product->image) {
                // Handle both old format (/public/storage/...) and new format (products/...)
                $oldImagePath = $product->image;
                if (strpos($oldImagePath, '/storage/') === 0 || strpos($oldImagePath, '/public/storage/') === 0) {
                    $oldImagePath = str_replace(['/storage/', '/public/storage/'], '', $oldImagePath);
                }
                Storage::disk('public')->delete($oldImagePath);
            }
            
            $imagePath = $request->file('image')->store('products', 'public');
            // Store relative path - Storage::url() will handle URL generation
            $validated['image'] = $imagePath;
        } else {
            unset($validated['image']);
        }

        // Handle additional images upload
        if ($request->hasFile('images')) {
            // Delete old additional images
            if ($product->images && is_array($product->images)) {
                foreach ($product->images as $oldImage) {
                    // Handle both old format (/public/storage/...) and new format (products/...)
                    $oldImagePath = $oldImage;
                    if (strpos($oldImagePath, '/storage/') === 0 || strpos($oldImagePath, '/public/storage/') === 0) {
                        $oldImagePath = str_replace(['/storage/', '/public/storage/'], '', $oldImagePath);
                    }
                    Storage::disk('public')->delete($oldImagePath);
                }
            }
            
            $imagePaths = [];
            foreach ($request->file('images') as $image) {
                $path = $image->store('products', 'public');
                $imagePaths[] = $path;
            }
            $validated['images'] = $imagePaths;
        } else {
            unset($validated['images']);
        }

        $validated['is_featured'] = $request->has('is_featured');
        $validated['is_active'] = $request->has('is_active');
        $validated['on_sale'] = $request->has('on_sale');
        
        // Set category_id to first category for backward compatibility
        $validated['category_id'] = $validated['categories'][0] ?? null;
        
        // Remove categories from validated array before updating product
        $categories = $validated['categories'];
        unset($validated['categories']);

        $product->update($validated);
        
        // Sync categories
        $product->categories()->sync($categories);

        return redirect()->route('admin.products.index')
            ->with('success', 'Product updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Product $product)
    {
        // Delete main image
        if ($product->image) {
            // Handle both old format (/public/storage/...) and new format (products/...)
            $imagePath = $product->image;
            if (strpos($imagePath, '/storage/') === 0 || strpos($imagePath, '/public/storage/') === 0) {
                $imagePath = str_replace(['/storage/', '/public/storage/'], '', $imagePath);
            }
            Storage::disk('public')->delete($imagePath);
        }

        // Delete additional images
        if ($product->images && is_array($product->images)) {
            foreach ($product->images as $image) {
                // Handle both old format (/public/storage/...) and new format (products/...)
                $imagePath = $image;
                if (strpos($imagePath, '/storage/') === 0 || strpos($imagePath, '/public/storage/') === 0) {
                    $imagePath = str_replace(['/storage/', '/public/storage/'], '', $imagePath);
                }
                Storage::disk('public')->delete($imagePath);
            }
        }

        $product->delete();

        return redirect()->route('admin.products.index')
            ->with('success', 'Product deleted successfully!');
    }
}
