<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AdminNotificationPreference;
use App\Models\Order;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class NotificationController extends Controller
{
    /**
     * Toggle browser notifications
     */
    public function toggle(Request $request)
    {
        $user = Auth::user();
        
        $preference = AdminNotificationPreference::firstOrCreate(
            ['user_id' => $user->id],
            [
                'browser_notifications_enabled' => false,
                'sound_enabled' => true,
                'new_order_notifications' => true,
                'new_user_notifications' => true,
            ]
        );

        $field = $request->input('field');
        $value = $request->input('value');

        if (in_array($field, ['browser_notifications_enabled', 'sound_enabled', 'new_order_notifications', 'new_user_notifications'])) {
            $preference->$field = $value;
            $preference->save();

            // If enabling browser notifications, request permission
            if ($field === 'browser_notifications_enabled' && $value) {
                return response()->json([
                    'success' => true,
                    'message' => 'Please allow browser notifications when prompted.',
                    'needs_permission' => true,
                ]);
            }

            return response()->json([
                'success' => true,
                'message' => 'Notification preference updated.',
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => 'Invalid field.',
        ], 400);
    }

    /**
     * Get notification preferences
     */
    public function preferences()
    {
        $user = Auth::user();
        
        $preference = AdminNotificationPreference::firstOrCreate(
            ['user_id' => $user->id],
            [
                'browser_notifications_enabled' => false,
                'sound_enabled' => true,
                'new_order_notifications' => true,
                'new_user_notifications' => true,
            ]
        );

        return response()->json([
            'success' => true,
            'preferences' => $preference,
        ]);
    }

    /**
     * Check for new notifications (polling endpoint)
     */
    public function check(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->is_admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized.',
            ], 403);
        }

        $preference = AdminNotificationPreference::where('user_id', $user->id)->first();
        
        if (!$preference || !$preference->browser_notifications_enabled) {
            return response()->json([
                'success' => true,
                'notifications' => [],
            ]);
        }

        $lastChecked = $preference->last_checked_at ?? now()->subHour();
        $notifications = [];

        // Check for new orders
        if ($preference->new_order_notifications) {
            $newOrders = Order::where('created_at', '>', $lastChecked)
                ->orderBy('created_at', 'desc')
                ->get();

            foreach ($newOrders as $order) {
                $notifications[] = [
                    'type' => 'new_order',
                    'title' => 'New Order Received',
                    'message' => "Order #{$order->order_number} for " . currency($order->total),
                    'icon' => '💰',
                    'url' => route('admin.orders.show', $order->id),
                    'timestamp' => $order->created_at->toIso8601String(),
                ];
            }
        }

        // Check for new users
        if ($preference->new_user_notifications) {
            $newUsers = User::where('created_at', '>', $lastChecked)
                ->where('is_admin', false)
                ->orderBy('created_at', 'desc')
                ->get();

            foreach ($newUsers as $newUser) {
                $notifications[] = [
                    'type' => 'new_user',
                    'title' => 'New Customer Sign Up',
                    'message' => "{$newUser->name} ({$newUser->email}) just signed up",
                    'icon' => '👤',
                    'url' => route('admin.users.show', $newUser->id),
                    'timestamp' => $newUser->created_at->toIso8601String(),
                ];
            }
        }

        // Update last checked time
        $preference->last_checked_at = now();
        $preference->save();

        return response()->json([
            'success' => true,
            'notifications' => $notifications,
        ]);
    }
}
