<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PageView;
use App\Models\Click;
use App\Models\UserSession;
use App\Models\Referrer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class AnalyticsController extends Controller
{
    public function index(Request $request)
    {
        $period = $request->get('period', '24h'); // Default: last 24 hours
        
        // Calculate start date based on period
        switch ($period) {
            case '1h':
                $startDate = now()->subHour();
                $timeUnit = 'hour';
                break;
            case '5h':
                $startDate = now()->subHours(5);
                $timeUnit = 'hour';
                break;
            case '24h':
                $startDate = now()->subHours(24);
                $timeUnit = 'hour';
                break;
            case '3d':
                $startDate = now()->subDays(3);
                $timeUnit = 'day';
                break;
            case '7d':
                $startDate = now()->subDays(7);
                $timeUnit = 'day';
                break;
            case '2w':
                $startDate = now()->subWeeks(2);
                $timeUnit = 'day';
                break;
            case '1m':
                $startDate = now()->subMonth();
                $timeUnit = 'day';
                break;
            default:
                $startDate = now()->subHours(24);
                $timeUnit = 'hour';
        }
        
        $endDate = now();

        // Overall Statistics
        $totalPageViews = PageView::whereBetween('viewed_at', [$startDate, $endDate])->count();
        $uniqueVisitors = UserSession::whereBetween('started_at', [$startDate, $endDate])->distinct('session_id')->count();
        $totalClicks = Click::whereBetween('clicked_at', [$startDate, $endDate])->count();
        $activeUsers = UserSession::where('is_active', true)
            ->where('last_activity_at', '>=', now()->subMinutes(5))
            ->count();

        // Page Views Over Time - Dynamic based on period
        $pageViewsData = [];
        if ($timeUnit === 'hour') {
            // For hourly periods, show data by hour
            $hours = $period === '1h' ? 1 : ($period === '5h' ? 5 : 24);
            $interval = $period === '1h' ? 1 : ($period === '5h' ? 1 : 1); // Show every hour
            
            for ($i = $hours - 1; $i >= 0; $i -= $interval) {
                $hourStart = now()->subHours($i);
                $hourEnd = $hourStart->copy()->addHour();
                $pageViewsData[] = [
                    'date' => $hourStart->format('H:00'),
                    'views' => PageView::whereBetween('viewed_at', [$hourStart, $hourEnd])->count(),
                    'visitors' => UserSession::whereBetween('started_at', [$hourStart, $hourEnd])->distinct('session_id')->count(),
                ];
            }
        } else {
            // For daily periods, show data by day
            $days = $period === '3d' ? 3 : ($period === '7d' ? 7 : ($period === '2w' ? 14 : 30));
            $interval = $period === '3d' ? 1 : ($period === '7d' ? 1 : ($period === '2w' ? 1 : 1)); // Show every day
            
            for ($i = $days - 1; $i >= 0; $i -= $interval) {
                $date = now()->subDays($i);
                $pageViewsData[] = [
                    'date' => $date->format('M d'),
                    'views' => PageView::whereDate('viewed_at', $date)->count(),
                    'visitors' => UserSession::whereDate('started_at', $date)->distinct('session_id')->count(),
                ];
            }
        }

        // Top Pages
        $topPages = PageView::select('path', DB::raw('COUNT(*) as views'))
            ->whereBetween('viewed_at', [$startDate, $endDate])
            ->groupBy('path')
            ->orderBy('views', 'desc')
            ->limit(10)
            ->get();

        // Traffic Sources
        $trafficSources = Referrer::select('source', DB::raw('SUM(visits) as total_visits'))
            ->whereBetween('last_seen_at', [$startDate, $endDate])
            ->groupBy('source')
            ->orderBy('total_visits', 'desc')
            ->get();

        // Referrers
        $topReferrers = Referrer::whereBetween('last_seen_at', [$startDate, $endDate])
            ->orderBy('visits', 'desc')
            ->limit(10)
            ->get();

        // Device Types
        $deviceTypes = PageView::select('device_type', DB::raw('COUNT(*) as count'))
            ->whereBetween('viewed_at', [$startDate, $endDate])
            ->groupBy('device_type')
            ->get();

        // Browsers
        $browsers = PageView::select('browser', DB::raw('COUNT(*) as count'))
            ->whereBetween('viewed_at', [$startDate, $endDate])
            ->whereNotNull('browser')
            ->groupBy('browser')
            ->orderBy('count', 'desc')
            ->limit(5)
            ->get();

        // Operating Systems
        $operatingSystems = PageView::select('os', DB::raw('COUNT(*) as count'))
            ->whereBetween('viewed_at', [$startDate, $endDate])
            ->whereNotNull('os')
            ->groupBy('os')
            ->orderBy('count', 'desc')
            ->limit(5)
            ->get();

        // Top Clicked Links
        $topClicks = Click::select('url', 'element_text', DB::raw('COUNT(*) as click_count'))
            ->whereBetween('clicked_at', [$startDate, $endDate])
            ->whereNotNull('url')
            ->groupBy('url', 'element_text')
            ->orderBy('click_count', 'desc')
            ->limit(20)
            ->get();

        // Active Users (Last 5 minutes)
        $recentActiveUsers = UserSession::where('is_active', true)
            ->where('last_activity_at', '>=', now()->subMinutes(5))
            ->with('user')
            ->orderBy('last_activity_at', 'desc')
            ->limit(20)
            ->get();

        // User Activity Timeline - Dynamic based on period
        $activityTimeline = [];
        if ($timeUnit === 'hour') {
            $hours = $period === '1h' ? 1 : ($period === '5h' ? 5 : 24);
            $interval = $period === '1h' ? 1 : ($period === '5h' ? 1 : 1);
            
            for ($i = $hours - 1; $i >= 0; $i -= $interval) {
                $hourStart = now()->subHours($i);
                $hourEnd = $hourStart->copy()->addHour();
                $activityTimeline[] = [
                    'hour' => $hourStart->format('H:00'),
                    'page_views' => PageView::whereBetween('viewed_at', [$hourStart, $hourEnd])->count(),
                    'clicks' => Click::whereBetween('clicked_at', [$hourStart, $hourEnd])->count(),
                ];
            }
        } else {
            // For daily periods, show data by day
            $days = $period === '3d' ? 3 : ($period === '7d' ? 7 : ($period === '2w' ? 14 : 30));
            
            if ($days <= 7) {
                // For 3-7 days, show hourly data
                $totalHours = $days * 24;
                $interval = $days <= 3 ? 1 : 3; // Show every hour for 3 days, every 3 hours for 7 days
                
                for ($i = $totalHours - 1; $i >= 0; $i -= $interval) {
                    $hourStart = now()->subHours($i);
                    $hourEnd = $hourStart->copy()->addHours($interval);
                    $activityTimeline[] = [
                        'hour' => $hourStart->format('M d H:00'),
                        'page_views' => PageView::whereBetween('viewed_at', [$hourStart, $hourEnd])->count(),
                        'clicks' => Click::whereBetween('clicked_at', [$hourStart, $hourEnd])->count(),
                    ];
                }
            } else {
                // For 2 weeks and 1 month, show daily data
                for ($i = $days - 1; $i >= 0; $i--) {
                    $dateStart = now()->subDays($i)->startOfDay();
                    $dateEnd = now()->subDays($i)->endOfDay();
                    $activityTimeline[] = [
                        'hour' => $dateStart->format('M d'),
                        'page_views' => PageView::whereBetween('viewed_at', [$dateStart, $dateEnd])->count(),
                        'clicks' => Click::whereBetween('clicked_at', [$dateStart, $dateEnd])->count(),
                    ];
                }
            }
        }

        // Geographic Data (if available)
        $countries = PageView::select('country', DB::raw('COUNT(*) as count'))
            ->whereBetween('viewed_at', [$startDate, $endDate])
            ->whereNotNull('country')
            ->groupBy('country')
            ->orderBy('count', 'desc')
            ->limit(10)
            ->get();

        // User vs Guest
        $userVsGuest = [
            'users' => PageView::whereBetween('viewed_at', [$startDate, $endDate])
                ->whereNotNull('user_id')
                ->distinct('user_id')
                ->count(),
            'guests' => UserSession::whereBetween('started_at', [$startDate, $endDate])
                ->whereNull('user_id')
                ->distinct('session_id')
                ->count(),
        ];

        // Average Session Duration
        $avgSessionDuration = UserSession::whereBetween('started_at', [$startDate, $endDate])
            ->whereNotNull('ended_at')
            ->avg('duration');

        return view('admin.analytics.index', compact(
            'totalPageViews',
            'uniqueVisitors',
            'totalClicks',
            'activeUsers',
            'pageViewsData',
            'topPages',
            'trafficSources',
            'topReferrers',
            'deviceTypes',
            'browsers',
            'operatingSystems',
            'topClicks',
            'recentActiveUsers',
            'activityTimeline',
            'countries',
            'userVsGuest',
            'avgSessionDuration',
            'period',
            'timeUnit'
        ));
    }

    public function trackClick(Request $request)
    {
        $request->validate([
            'element_type' => 'required|string',
            'element_id' => 'nullable|string',
            'element_class' => 'nullable|string',
            'element_text' => 'nullable|string',
            'url' => 'nullable|string',
            'page_url' => 'required|string',
        ]);

        Click::create([
            'user_id' => auth()->id(),
            'session_id' => session()->getId(),
            'element_type' => $request->element_type,
            'element_id' => $request->element_id,
            'element_class' => $request->element_class,
            'element_text' => $request->element_text,
            'url' => $request->url,
            'page_url' => $request->page_url,
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'clicked_at' => now(),
        ]);

        // Update session click count
        $session = UserSession::where('session_id', session()->getId())->first();
        if ($session) {
            $session->increment('clicks');
            $session->update(['last_activity_at' => now()]);
        }

        return response()->json(['success' => true]);
    }

    public function getActiveUsers()
    {
        $activeUsers = UserSession::where('is_active', true)
            ->where('last_activity_at', '>=', now()->subMinutes(5))
            ->with('user')
            ->orderBy('last_activity_at', 'desc')
            ->limit(50)
            ->get()
            ->map(function ($session) {
                return [
                    'id' => $session->id,
                    'user' => $session->user ? $session->user->name : 'Guest',
                    'page' => $session->landing_page,
                    'device' => $session->device_type,
                    'browser' => $session->browser,
                    'last_activity' => $session->last_activity_at->diffForHumans(),
                ];
            });

        return response()->json([
            'count' => $activeUsers->count(),
            'users' => $activeUsers,
        ]);
    }
}
